/*
 * Charles Copyright (c) 2018-2021, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <unistd.h>
#include <sys/stat.h>

#include "formatter.h"
#include "epub.h"
#include "text.h"

int LEN = _LEN, WIDTH = _WIDTH, LENGTH = _LENGTH, HEAD = _HEAD, LINES = _LINES,
    CHARS = _CHARS, HALF = _HALF, RTF_HALF = _RTF_HALF, TOP = _TOP, RTF_TOP = _RTF_TOP,
    LEFT = _LEFT, RTF_LEFT = _RTF_LEFT, ORPHAN_CHAPTER = _ORPHAN_CHAPTER, ORPHAN_BODY = _ORPHAN_BODY;

void *memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
      fprintf( stderr, "malloc(): %s", strerror( errno ));

   return ptr;
}

char *str_dup( char *str, int len )
{
   char *ptr, *ptr2;

   if ( str == NULL )
      return NULL;

   if ( len < 0 )
      for( len = 0, ptr = str; *ptr; ++ptr )
         ++len;

   if (( ptr = memory( len + 1 )) == NULL )
      return NULL;

   ptr[ len ] = '\0';

   for( ptr2 = ptr; len; --len )
      *ptr2++ = *str++;

   return ptr;
}

void print_pdf( int newpage )
{
   struct meta *meta;
   struct tsml *tree;
   struct para *paras;
   struct stack *pages, *pdf;
   int n;

   if (( tree = make_tsml_tree( stdin )) == NULL )
      return;

   if (( meta = make_meta()) == NULL )
   {
      free_tsml_tree( tree );
      return;
   }

   if (( paras = make_para_list( tree, meta, 1 )) == NULL )
   {
      free_meta( meta );
      free_tsml_tree( tree );
      return;
   }

   free_tsml_tree( tree );

   if (( pages = make_page_stack( paras, newpage )) == NULL )
   {
      free_meta( meta );
      free_para_list( paras );
      return;
   }

   free_para_list( paras );

   if (( pdf = make_pdf( pages, meta )) == NULL )
   {
      free_meta( meta );
      free_page_stack( pages );
      return;
   }

   free_meta( meta );
   free_page_stack( pages );

   for( n = 0; n < pdf->used; n += 2 )
   {
      fwrite( pdf->values[ n + 1 ].ptr, 1, pdf->values[ n ].num, stdout );
      free( pdf->values[ n + 1 ].ptr );
   }

   stack_free( pdf );
}

void redirect_stdin( char *infile )
{
   FILE *file;

   if ( infile == NULL || ( *infile == '-' && ! infile[ 1 ] ))
      return;

   if (( file = fopen( infile, "r" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"r\" ): %s", infile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdin );
   stdin = file;
}

void redirect_stdout( char *outfile )
{
   FILE *out;

   if ( outfile == NULL )
      return;

   unlink( outfile );
   if (( out = fopen( outfile, "w" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"w\" ): %s\n", outfile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdout );
   stdout = out;
}

void set_constants( int a4 )
{
   if ( a4 )
   {
      LEN    = _A4_LEN;
      WIDTH  = _A4_WIDTH;
      LENGTH = _A4_LENGTH;
      HEAD   = _A4_HEAD;
      LINES  = _A4_LINES;
      CHARS  = _A4_CHARS;
      HALF   = _A4_HALF;

      RTF_HALF = _A4_RTF_HALF;
      TOP      = _A4_TOP;
      RTF_TOP  = _A4_RTF_TOP;
      LEFT     = _A4_LEFT;
      RTF_LEFT = _A4_RTF_LEFT;

      ORPHAN_CHAPTER = _A4_ORPHAN_CHAPTER;
      ORPHAN_BODY    = _A4_ORPHAN_BODY;
   }
}

int main( int argc, char **argv )
{
   char *dir = ".";
   int opt, args = 0, newpage = 0, convert = 0, export = 0, 
            epub = 0, a4 = 0, cover = 0;

   while(( opt = getopt( argc, argv, "iaxcne" )) != -1 )
   {
      ++args;

      switch( opt )
      {
         case 'a':
            ++a4;
            break;

         case 'c':
            ++convert;
            break;

         case 'e':
            ++epub;
            break;
         
         case 'i':
            ++cover;
            break;
         
         case 'n':
            ++newpage;
            break;

         case 'x':
            ++export;
            break;
      }
   }

   opt = convert + export + epub;

   if ( opt > 1 )
   {
      fputs( "-c, -e, and -x, are mutually exclusive\n", stderr );
      exit( 1 );
   }
   
   argc -= args;
   argv += args;

   if ( argc > 1 )
      redirect_stdin( argv[ 1 ] );

   if ( argc > 2 )
   {
      if ( epub )
         dir = argv[ 2 ];
      else
         redirect_stdout( argv[ 2 ] );
   }

   set_constants( a4 );

   if ( convert )
      print_tsml();
   else if ( export )
      print_rtf( newpage );
   else if ( epub )
      make_ebook( dir, cover );
   else
      print_pdf( newpage );

   return 0;
}
