/*
 * Charles Copyright (c) 2018-2021, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "formatter.h"

#include <dirent.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>

extern void *memory( int );
extern char *str_dup( char *, int );

extern int LEN, WIDTH, LENGTH, HEAD, LINES, CHARS, HALF, RTF_HALF, TOP, RTF_TOP,
           LEFT, RTF_LEFT, ORPHAN_CHAPTER, ORPHAN_BODY;

int epub_normalize_whitespace( char *input, struct string *source )
{
   int len, ignore, last, space, underscore = 0, hyphen = 0;
   char *ptr;

   for( ptr = input; *ptr; ++ptr )
      if ( ! isspace( *ptr ))
         break;

   for( ignore = 0, space = 1, last = 0; *ptr; ++ptr )
   {
      if ( ! isspace( *ptr ))
      {
         last = 0;

         if ( space )
            ++space;

         if ( *ptr == '\\' && ( len = process_accent(( unsigned char *)ptr + 1, source )) > 0 )
         {
            ptr += len;
            continue;
         }
      }
      else if ( last )
         continue;
      else
      {
         ++last;
         ++space;
         *ptr = ' ';
      }

      switch( *ptr )
      {
         case '_':
            if ( !underscore && space == 2 )
               ++ignore;

            underscore ^= 1;
            if ( string_concat( source, ( underscore ? "<em>" : "</em>" )))
               return 1;
            break;

         case '-':
            if ( ++hyphen == 2 && source->used )
            {
               --source->top;
               ++source->free;
               --source->used;
               *source->top = '\0';

               if ( string_concat( source, "&#8212;" ))
                  return 1;

               hyphen = 0;
            }
            else if ( string_append( source, *ptr ))
               return 1;
            break;

         case '&':
            if ( string_concat( source, "&amp;" ))
               return 1;
            break;

         case '"':
            if ( string_concat( source, ( space || ignore ? "&#8220;" : "&#8221;" )))
               return 1;
            break;

         case '\'':
            if ( string_concat( source, ( space || ignore ? "&#8216;" : "&#8217;" )))
               return 1;
            break;

         case '`':
            if ( string_concat( source, "&#8217;" ))
               return 1;
            break;

         case '<':
            if ( string_concat( source, "&lt;" ))
               return 1;
            break;

         case '>':
            if ( string_concat( source, "&gt;" ))
               return 1;
            break;

         case '|':
            if ( string_concat( source, "<br/>" ))
               return 1;
            break;

         default:
            if ( string_append( source, *ptr ))
               return 1;
      }

      if ( *ptr != '-' )
         hyphen = 0;

      if ( space > 1 )
         space = 0;

      if ( ignore == 1 )
         ++ignore;
      else if ( ignore > 1 )
         ignore = 0;
   }

   return 0;
}

char *meta_escape( char *input )
{
   char *ptr;
   struct string *output;

   if (( output = string_make( NULL )) == NULL )
      exit( 1 );

   if ( input == NULL )
   {
      if ( string_concat( output, "UNKNOWN" ))
         exit( 1 );

      goto OK;
   }

   for( ptr = input; *ptr; ++ptr )
      if ( *ptr == ';' )
         *ptr = ' ';

   if ( epub_normalize_whitespace( input, output ))
      exit( 1 );

OK:
   ptr = output->str;
   free( output );
   return ptr;
}

struct stack *epub_wrap_lines( char *input )
{
   struct stack *para;
   struct string *line = NULL, *source = NULL;
   union stack_val val;
   unsigned char *ptr;
   int i, n, first = 1;

   for( n = 0, ptr = ( unsigned char *)input; *ptr; ++ptr )
      if ( *ptr == '_' )
         ++n;

   if ( n% 2 )
   {
      fprintf( stderr, "unbalanced underscores in: %s\n", input );
      exit( 1 );
   }

   if ( input == NULL || ( para = stack_make()) == NULL )
      return NULL;

   if (( source = string_make( NULL )) == NULL )
      goto ERR;

   if ( epub_normalize_whitespace( input, source ))
      goto ERR;

   if (( line = string_make( NULL )) == NULL )
      goto ERR;

   for( n = i = 0; source->str[ i ]; )
   {
      /*
       * Remove leading whitespace.
       */

      for( ptr = ( unsigned char *)&source->str[ i ]; *ptr; ++ptr, ++i )
         if ( *ptr != ' ' )
            break;

      for( n = ( first ? n : 0 ), ptr = ( unsigned char *)&source->str[ i ]; n < LEN && *ptr; ++ptr, ++i )
      {
         /*
          * Do not count underscores.  They will be removed.
          */

         if (( *ptr & 0xC0 ) != 0x80 && *ptr != '_' )
            ++n;

         if ( string_append( line, *ptr ))
            goto ERR;
      }

      /*
       * If we have not broken the paragraph at whitespace naturally,
       * move the break back until we do break at whitespace.
       */

      while( line->used && *ptr && *( ptr - 1 ) != ' ' && *ptr != ' ' )
      {
         --line->used;
         ++line->free;
         --line->top;

         line->str[ line->used ] = '\0';
         --ptr;
         --i;
      }

      /*
       * Remove trailing whitespace from the new line.
       */

      while( line->used && *--ptr == ' ' )
      {
         --line->used;
         ++line->free;
         --line->top;

         line->str[ line->used ] = '\0';
         --i;
      }

      /*
       * If we did not find whitespace to break at, we have a line with a
       * token in it that is longer than the line length.  We create an
       * overlong line by moving the break forward until we find whitespace
       * or we run out of data.
       */

      if ( ! line->used )
         for( ptr = ( unsigned char *)&source->str[ i ]; *ptr && *ptr != ' '; ++ptr, ++i )
            if ( string_append( line, *ptr ))
               goto ERR;

      /*
       * Do not create blank lines.  This can happen if input is all
       * whitespace.
       */

      if ( line->used )
      {
         if (( val.ptr = str_dup( line->str, line->used )) == NULL )
            goto ERR;

         if ( stack_push( para, val ))
         {
            free( val.ptr );
            goto ERR;
         }
      }

      string_clear( line );
      first = 0;
   }

   string_free( source );
   string_free( line );
   return para;

ERR:
   if ( source != NULL )
      string_free( source );

   if ( line != NULL )
      string_free( line );

   while( para->used )
   {
      union stack_val *vptr = stack_pop( para );
      free( vptr->ptr );
   }

   stack_free( para );
   return NULL;
}

struct para *epub_make_para( char *data, enum paras type, struct meta *meta, int restart )
{
   struct para *pptr;
   char *new = NULL;

   if (( pptr = memory( sizeof( struct para ))) == NULL )
      return NULL;

   if ( meta != NULL )
   {
      if ( type == PART )
      {
         ++meta->part;
         meta->section = 0;
         meta->chapter = 0;

         if (( new = substitute_no( data, meta->part, meta->section, meta->chapter )) == NULL )
         {
            free( pptr );
            return NULL;
         }
      }
      else if ( type == SECTION )
      {
         ++meta->section;
         meta->chapter = 0;

         if (( new = substitute_no( data, meta->part, meta->section, meta->chapter )) == NULL )
         {
            free( pptr );
            return NULL;
         }
      }
      else if ( type == CHAPTER )
      {
         ++meta->chapter;

         if (( new = substitute_no( data, meta->part, meta->section, meta->chapter )) == NULL )
         {
            free( pptr );
            return NULL;
         }
      }
   }

   pptr->type = type;

   if (( pptr->lines = epub_wrap_lines(( new == NULL ? data : new ))) == NULL )
   {
      if ( new != NULL )
         free( new );

      free( pptr );
      return NULL;
   }

   if ( new != NULL )
      free( new );

   pptr->next = NULL;
   return pptr;
}

struct para *epub_make_para_list( struct tsml *tree, struct meta *meta, int restart )
{
   struct tsml *tptr;
   struct para *list = NULL, *pptr = NULL, **prev = NULL;
   enum paras type;

   if ( tree == NULL || tree->type != ROOT )
      return NULL;

   /*
    * If the y element is present, make it root.
    */

   if ( tree->children            != NULL && tree->children->type == OPEN &&
        tree->children->data      != NULL &&
        tree->children->data[ 0 ] == 'y'  && ! tree->children->data[ 1 ] )
      tree = tree->children;

   for( tptr = tree->children; tptr != NULL; tptr = tptr->next )
   {
      /*
       * Skip whitespace outside of elements.
       */

      if ( tptr->type == DATA )
         continue;

      if ( tptr->data == NULL )
         goto ERR;

      /*
       * Bail if the name of this element is more than 1 character in
       * length or if the element has no children.
       */

      if (( *tptr->data && *( tptr->data + 1 )) || tptr->children == NULL )
         break;

      switch( *tptr->data )
      {
         case 'p':
            type = PART;
            break;

         case 's':
            type = SECTION;
            break;

         case 'c':
            type = CHAPTER;
            break;

         case 'b':
            type = BODY;
            break;

         case 'e':
            type = EMPHASIZED;
            break;

         case 'i':
            type = INTERRUPT;
            break;

         case 'm':
            if ( meta != NULL )
               extract_metadata( tptr, meta );
            continue;

         default:
            goto ERR;
      }

      /*
       * Bail if the element has more than one child or if the lone child
       * is not a data node.
       */

      if ( tptr->children->next != NULL || tptr->children->type != DATA ||
           ( pptr = epub_make_para( tptr->children->data, type, meta, restart )) == NULL )
         break;

      if ( list == NULL )
         list = pptr;

      if ( prev != NULL )
         *prev = pptr;

      prev = &pptr->next;
   }

ERR:
   if ( tptr != NULL )
   {
      free_para_list( list );
      list = NULL;
   }

   return list;
}

void make_dir( char *path )
{
   int mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;

   if ( mkdir( path, mode ) < 0 )
   {
      fprintf( stderr, "mkdir( %s ): %s\n", path, strerror( errno ));
      exit( 1 );
   }
}

FILE *open_file( char *path )
{
   FILE *file;

   if (( file = fopen( path, "w" )) == NULL )
   {
      fprintf( stderr, "fopen( %s ): %s\n", path, strerror( errno ));
      exit( 1 );
   }

   return file;
}

void put_string( char *str, FILE *file )
{
   if ( fputs( str, file ) < 0 )
   {
      fprintf( stderr, "fputs( %s ): %s", str, strerror( errno ));
      exit( 1 );
   }
}

void concat( struct string *str, char *data )
{
   if ( string_concat( str, data ))
      exit( 1 );
}

void remove_dir( char *name )
{
   DIR *dir;
   struct dirent entry, *result;
   char buffer[ MAXPATHLEN + 1 ];

   if (( dir = opendir( name )) == NULL )
   {
      if ( errno == ENOENT )
         return;

      fprintf( stderr, "opendir( %s ): %s\n", name, strerror( errno ));
      exit( 1 );
   }

   readdir_r( dir, &entry, &result );
   readdir_r( dir, &entry, &result );

   while( ! readdir_r( dir, &entry, &result ) && result != NULL )
   {
      snprintf( buffer, sizeof( buffer ), "%s/%s", name, entry.d_name );

      if ( unlink( buffer ) < 0 )
      {
         fprintf( stderr, "unlink( %s ): %s\n", buffer, strerror( errno ));
         exit( 1 );
      }
   }

   closedir( dir );

   if ( rmdir( name ) < 0 )
   {
      fprintf( stderr, "rmdir( %s ): %s", name, strerror( errno ));
      exit( 1 );
   }
}

void remove_ebook( char *root )
{
   char **ptr, buffer[ MAXPATHLEN + 1 ], *dirs[] = { "ebook/META-INF", "ebook/OEBPS", "ebook", NULL };

   for( ptr = dirs; *ptr != NULL; ++ptr )
   {
      snprintf( buffer, sizeof( buffer ), "%s/%s", root, *ptr );
      remove_dir( buffer );
   }
}

void make_dirs( char *root )
{
   char buffer[ MAXPATHLEN + 1 ],
        *mime_type = "application/epub+zip",

        *content_xml = "<?xml version=\"1.0\"?>\n"
                       "<container version=\"1.0\" xmlns=\"urn:oasis:names:tc:opendocument:xmlns:container\">\n"
                       "<rootfiles>\n"
                       "<rootfile full-path=\"OEBPS/content.opf\" media-type=\"application/oebps-package+xml\"/>\n"
                       "</rootfiles>\n"
                       "</container>",

        *styles = "h1 { font-size: 2em; font-weight: bold; text-align: center; margin: 1em; margin-left: 0; margin-right: 0; }\n"
                  "h2 { font-size: 1.5em; font-weight: bold; text-align: center; margin: 1em; margin-left: 0; margin-right: 0; }\n"
                  "h3 { font-size: 1.25em; font-weight: bold; text-align: center; margin: 1em; margin-left: 0; margin-right: 0; }\n"
                  "p { margin: 0; padding: 0; }\n"
                  "p.copy { text-indent: 0; font-size: 0.8em; text-align: center; margin-top: 4em; }\n"
                  "p.break { text-indent: 0; text-align: center; }\n"
                  "p.body { text-indent: 1.5em; }\n"
                  "p.first { text-indent: 0; }\n"
                  "p.emph { text-indent: 1.5em; font-style: italic;}\n"
                  "p.first_emph { text-indent: 0; font-style: italic;}\n"
                  "em { font-style: italic; }\n"
                  "p.emph em, p.first_emph em { font-weight: 900; }\n";

   FILE *file;

   remove_ebook( root );
   snprintf( buffer, sizeof( buffer ), "%s/ebook", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/META-INF", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/mimetype", root );
   file = open_file( buffer );
   put_string( mime_type, file );
   fclose( file );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/META-INF/container.xml", root );
   file = open_file( buffer );
   put_string( content_xml, file );
   fclose( file );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/stylesheet.css", root );
   file = open_file( buffer );
   put_string( styles, file );
   fclose( file );
}

char *epub_make_date()
{
   struct tm lt;
   time_t t;
   static char date[ 64 ];

   t = time( NULL );
   localtime_r( &t, &lt );
   strftime( date, sizeof( date ), "%Y-%m-%dT%H:%M:%SZ", &lt );

   return date;
}

char *make_identifier( struct meta *meta )
{
   char *title, *author, *ptr;
   struct string *id;

   if (( id = string_make( NULL )) == NULL )
      return NULL;

   title = meta_escape( meta->title );
   author = meta_escape( meta->author );

   for( ptr = title; *ptr; ++ptr )
      if ( *ptr == ' ' )
         *ptr = '-';

   for( ptr = author; *ptr; ++ptr )
      if ( *ptr == ' ' )
         *ptr = '-';

   concat( id, title );
   concat( id, "-" );
   concat( id, author );
   concat( id, "-" );

   concat( id, epub_make_date());
   free( title );
   free( author );

   ptr = id->str;
   free( id );
   return ptr;
}

void write_toc( char *root, struct string *toc )
{
   FILE *file;
   char buffer[ MAXPATHLEN + 1 ],
        *preamble = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                    "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:epub=\"http://www.idpf.org/2007/ops\">\n"
                    "<head>\n"
                    "<title>toc.xhtml</title>\n"
                    "<link href=\"stylesheet.css\" rel=\"stylesheet\" type=\"text/css\" />\n"
                    "</head>\n"
                    "<body>\n"
                    "<nav id=\"toc\" epub:type=\"toc\">\n"
                    "<h1>Contents</h1>\n"
                    "<ol>\n";

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/toc.xhtml", root );
   file = open_file( buffer );

   put_string( preamble, file );
   put_string( toc->str, file );
   put_string( "</ol>\n</nav>\n</body>\n</html>\n", file );

   fclose( file );
}

void add_preamble( FILE *file, int fileno )
{
   char buffer[ 1024 ],
        *preamble = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n"
                    "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:epub=\"http://www.idpf.org/2007/ops\">\n"
                    "<head>\n"
                    "<title>file%d.xhtml</title>\n"
                    "<link href=\"stylesheet.css\" rel=\"stylesheet\" type=\"text/css\" />\n"
                    "</head>\n"
                    "<body>\n";

   snprintf( buffer, sizeof( buffer ), preamble, fileno );
   put_string( buffer, file );
}

void add_closer( FILE *file )
{
   put_string( "</body>\n</html>\n", file );
}

void advance_file( FILE **file, int *fileno, char *root, struct string *data, struct string *spine, struct string *toc )
{
   char buffer[ 1024 ];

   if ( *file != NULL )
   {
      ++*fileno;
      add_closer( *file );
      fclose( *file );
   }

   if ( toc != NULL )
   {
      snprintf( buffer, sizeof( buffer ), "<li><a href=\"file%d.xhtml\">", *fileno );
      concat( toc, buffer );
   }

   snprintf( buffer, sizeof( buffer ), "<itemref idref=\"item%d\" />\n", *fileno );
   concat( spine, buffer );

   snprintf( buffer, sizeof( buffer ),
             "<item id=\"item%d\" href=\"file%d.xhtml\" media-type=\"application/xhtml+xml\" />\n",
             *fileno, *fileno );
   concat( data, buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/file%d.xhtml", root, *fileno );
   *file = open_file( buffer );
   add_preamble( *file, *fileno );
}

void epub_add_title_page( struct meta *meta, FILE **file, int *fileno, char *root,
                          struct string *data, struct string *spine, struct string *toc )
{
   char *tptr = NULL, *aptr = NULL;
   struct string *item;

   advance_file( file, fileno, root, data, spine, NULL );

   if (( item = string_make( NULL )) == NULL )
      exit( 1 );

   concat( item, "<h1>" );
   concat( item, tptr = meta_escape( meta->title ));
   concat( item, "</h1>\n<h2>" );
   concat( item, aptr = meta_escape( meta->author ));
   concat( item, "</h2>\n" );

   free( tptr );
   free( aptr );
   put_string( item->str, *file );

   if ( meta->copyright == NULL )
      goto END;

   string_clear( item );
   concat( item, "<p class=\"copy\">Copyright &#169; " );
   tptr = meta_escape( meta->copyright );
   concat( item, tptr );
   free( tptr );
   concat( item, "</p>\n" );

   put_string( item->str, *file );

END:
   string_free( item);
   advance_file( file, fileno, root, data, spine, toc );
}

void add_files( char *root, struct meta *meta, struct para *paras, struct string *data, int cover )
{
   union stack_val val;
   FILE *file = NULL;
   struct string *toc, *spine;
   struct para *ptr;
   int fileno = 0, last = -1, no_first_header = 0;

   if (( toc = string_make( NULL )) == NULL || ( spine = string_make( NULL )) == NULL )
      exit( 1 );

   if ( cover )
   {
      advance_file( &file, &fileno, root, data, spine, NULL );
      put_string( "<p class=\"break\"><img src=\"cover2.png\" alt=\"Cover Image\" /></p>\n", file );
   }

   epub_add_title_page( meta, &file, &fileno, root, data, spine, toc );

   for( ptr = paras; ptr != NULL; ptr = ptr->next )
   {
      switch( ptr->type )
      {
         case PART:
         case SECTION:
         case CHAPTER:
            if ( ptr != paras )
            {
               if ( no_first_header )
               {
                  concat( toc, "PREFACE</a></li>\n" );
                  no_first_header = 0;
               }

               advance_file( &file, &fileno, root, data, spine, toc );
            }

            put_string(( ptr->type == PART ? "<h1>" : ( ptr->type == SECTION ? "<h2>" : "<h3>" )), file );

            while( ptr->lines->used )
            {
               stack_shift( ptr->lines, &val );
               put_string( val.ptr, file );
               put_string( " ", file );

               concat( toc, val.ptr );
               free( val.ptr );
            }

            concat( toc, "</a></li>\n" );
            put_string(( ptr->type == PART ? "</h1>\n" : ( ptr->type == SECTION ? "</h2>\n" : "</h3>\n" )), file );
            break;

         case INTERRUPT:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"break\">&#8212;&#8212;</p>\n", file );
            break;

         case BODY:
         case EMPHASIZED:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );

            if ( ptr->type == BODY )
               put_string(( last != BODY && last != EMPHASIZED ? "first\">\n" : "body\">\n " ), file );
            else
               put_string(( last != BODY && last != EMPHASIZED ? "first_emph\">\n" : "emph\">\n " ), file );

            while( ptr->lines->used )
            {
               stack_shift( ptr->lines, &val );
               put_string( val.ptr, file );
               put_string( "\n", file );
               free( val.ptr );
            }

            put_string( "</p>\n", file );
      }

      last = ptr->type;
   }

   add_closer( file );
   fclose( file );

   if ( no_first_header )
      concat( toc, "PREFACE</a></li>\n" );

   write_toc( root, toc );
   string_free( toc );

   concat( data, "</manifest>\n<spine>\n" );
   concat( data, spine->str );
   concat( data, "</spine>\n" );

   string_free( spine );
}

void add_content( char *root, struct meta *meta, struct para *paras, int cover )
{
   FILE *file;
   struct string *data;
   char buffer[ MAXPATHLEN + 1 ], *ptr,

        *preamble = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
                    "<package xmlns=\"http://www.idpf.org/2007/opf\" "
                             "xmlns:dc=\"http://purl.org/dc/elements/1.1/\" "
                             "unique-identifier=\"book-id\" "
                             "version=\"3.0\">\n"
                    "<metadata>\n",

        *manifest = "<manifest>\n"
                    "<item id=\"toc\" properties=\"nav\" href=\"toc.xhtml\" media-type=\"application/xhtml+xml\" />\n"
                    "<item id=\"css\" href=\"stylesheet.css\" media-type=\"text/css\" />\n";

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/content.opf", root );

   file = open_file( buffer );
   put_string( preamble, file );

   if (( data = string_make( NULL )) == NULL )
      exit( 1 );

   concat( data, "<dc:title>" );
   ptr = meta_escape( meta->title );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:title>\n" );

   concat( data, "<dc:creator>" );
   ptr = meta_escape( meta->author );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:creator>\n");

   concat( data, "<dc:identifier id=\"book-id\">" );
   ptr = make_identifier( meta );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:identifier>\n" );

   snprintf( buffer, sizeof( buffer ), "<meta property=\"dcterms:modified\">%s</meta>\n",
             epub_make_date());
   concat( data, buffer );

   concat( data, "<dc:language>" );
   concat( data, ( meta->language == NULL ? "en" : meta->language ));
   concat( data, "</dc:language>\n</metadata>\n" );

   concat( data, manifest );

   if ( cover )
   {
      concat( data, "<item id=\"cover1\" properties=\"cover-image\" href=\"cover1.png\" media-type=\"image/png\" />\n" );
      concat( data, "<item id=\"cover2\" href=\"cover2.png\" media-type=\"image/png\" />\n" );
   }

   add_files( root, meta, paras, data, cover );
   concat( data, "</package>\n" );

   put_string( data->str, file );
   string_free( data );
   fclose( file );
}

void make_ebook( char *root, int restart, int cover )
{
   struct meta *meta;
   struct tsml *tree;
   struct para *paras;

   if (( tree = make_tsml_tree( stdin )) == NULL )
      return;

   if (( meta = make_meta()) == NULL )
   {
      free_tsml_tree( tree );
      return;
   }

   if (( paras = epub_make_para_list( tree, meta, restart )) == NULL )
   {
      free_meta( meta );
      free_tsml_tree( tree );
      return;
   }

   free_tsml_tree( tree );

   make_dirs( root );
   add_content( root, meta, paras, cover );

   free_meta( meta );
   free_para_list( paras );
   return;
}
