/*
 * Charles Copyright (c) 2018-2023, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "formatter.h"

extern int LEN, WIDTH, LENGTH, HEAD, LINES, CHARS, HALF, RTF_HALF, TOP, RTF_TOP,
           LEFT, RTF_LEFT, ORPHAN_CHAPTER, ORPHAN_BODY;

int get_line( struct string *item, int spaces )
{
   int c;

   string_clear( item );

   while(( c = fgetc( stdin )) != EOF )
   {
      if ( c == '\r' )
         continue;

      if ( c == '\n' )
         break;

      if (( c == '[' || c == ']' || c == '\\' ) && string_append( item, '\\' ))
         exit( 1 );

      if ( isspace( c ))
         c = ' ';

      if ( string_append( item, ( char )c ))
         exit( 1 );
   }

   if ( c == EOF && ! item->used )
      return 1;

   if ( ! spaces )
      while( item->used && item->str[ item->used - 1 ] == ' ' )
      {
         --item->used;
         ++item->free;
         --item->top;
         *item->top = '\0';
      }

   return 0;
}

int compare( char *ptr1, char *ptr2 )
{
   char *ptr;

   for( ptr = ptr2; *ptr; ++ptr )
      *ptr = tolower( *ptr );

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      return 1;

   return 0;
}

int is_blank( char *item )
{
   char *ptr;

   for( ptr = item; *ptr; ++ptr )
      if ( ! isspace( *ptr ))
         return 0;

   return 1;
}

void replace_semicolons( char *item )
{
   char *ptr;

   for( ptr = item; *ptr; ++ptr )
      if ( *ptr == ';' )
         *ptr = '~';
}

int print_title_page( struct string *item )
{
   char *ptr;
   int first = 1, multi, found = 0;

   fputs( "[m[\n", stdout );

   for( multi = 0; ; )
   {
      if ( get_line( item, 1 ))
         break;

      if ( !item->used )
      {
         if ( first )
            continue;
         else
            break;
      }

      first = 0;

      if ( multi )
      {
         if ( isspace( *item->str ))
         {
            replace_semicolons( item->str );
            printf( "%s%s", ( multi > 1 ? ";" : "" ), item->str );
            ++multi;
            continue;
         }
         else
         {
            fputs( "]]\n", stdout );
            multi = 0;
         }
      }

      for( ptr = item->str; *ptr; ++ptr )
         if ( *ptr == ':' )
         {
            ++found;
            *ptr = '\0';

            if ( compare( "title",      item->str ) || compare( "author",   item->str ) ||
                 compare( "revision",   item->str ) || compare( "contact",  item->str ) ||
                 compare( "header",     item->str ) || compare( "language", item->str ) ||
                 compare( "watermark",  item->str ) || compare( "copyright",  item->str ))
            {
               char c;

               switch( *item->str )
               {
                  case 't':
                     c = 't';
                     break;

                  case 'a':
                     c = 'a';
                     break;

                  case 'r':
                     c = 'r';
                     break;

                  case 'c':
                     c = ( item->str[ 2 ] == 'n' ? 'c' : 'y' );
                     break;

                  case 'h':
                     c = 'h';
                     break;

                  case 'l':
                     c = 'l';
                     break;

                  case 'w':
                     c = 'w';
                     break;

                  default:
                     c = '\0';
               }

               if ( ! is_blank( ++ptr ))
               {
                  replace_semicolons( ptr );
                  printf( "   [%c[%s]]\n", c, ptr );
               }
               else
               {
                  printf( "   [%c[", c );
                  ++multi;
               }
            }

            break;
         }

      if ( ! found )
         break;
   }

   if ( multi )
      fputs( "]]\n", stdout );

   fputs( "]m]\n\n", stdout );
   return found;
}

int is_note( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->used > 3 && item->str[ 0 ] == '\\' && item->str[ 1 ] == '['
                       && item->str[ 2 ] == '\\' && item->str[ 3 ] == '[' )
   {
      do
         if ( item->used > 3 && item->str[ item->used - 1 ] == ']' && item->str[ item->used - 2 ] == '\\'
                             && item->str[ item->used - 3 ] == ']' && item->str[ item->used - 4 ] == '\\' )
            break;
      while( ! get_line( item, 0 ));

      return 1;
   }

   return 0;
}

void print_paras( struct string *item, int found )
{
   struct string *para;
   int c, eof;

   if (( para = string_make( NULL )) == NULL )
      return;

   for( eof = ( found ? get_line( item, 0 ) : 0 ); ! eof; eof = get_line( item, 0 ))
   {
      if ( ! para->used && item->used )
      {
         switch( *item->str )
         {
            case '*':
               c = 'p';
               break;
            
            case '=':
               c = 's';
               break;

            case '.':
               c = 'c';
               break;

            case '#':
               c = 'i';
               break;

            case '%':
               c = 'e';
               break;

            default:
               c = 'b';
         }
      }

      if ( item->used )
      {
         if ( is_note( item ))
            continue;

         if ( para->used && string_append( para, ' ' ))
            break;

         if ( string_merge( para, item ))
            break;

         continue;
      }

      if ( ! para->used )
         continue;

      printf( "[%c[%s]]\n\n", c, ( c == 'p' || c == 'c' || c == 's' || c == 'e' ? &para->str[ 1 ] : ( c == 'i' ? "#" : para->str )));
      string_clear( para );
   }

   if ( para->used )
      printf( "[%c[%s]]\n\n", c, ( c == 'p' || c == 'c' || c == 's' || c == 'e' ? &para->str[ 1 ] : ( c == 'i' ? "#" : para->str )));

   string_free( para );
}

void print_tsml()
{
   struct string *item;
   int found;

   if (( item = string_make( NULL )) == NULL )
      exit( 1 );

   fputs( "[y[\n", stdout );

   found = print_title_page( item );
   print_paras( item, found );

   fputs( "]y]\n", stdout );
   string_free( item );
}

int add_header_line( struct stack *lines, char *header )
{
   union stack_val val;

   if (( val.ptr = str_dup( header, -1 )) == NULL )
      return 1;

   if ( stack_push( lines, val ))
   {
      free( val.ptr );
      return 1;
   }

   return 0;
}

/*
 * With unbalanced underscores on a line, we generate bad RTF, so we must add
 * an extra brace at the end of the line if underline != 0.

 * This is not a problem when producing EPUBs or PDFs.  The paragraph
 * wrappers for all output types detect odd quanties of underscores per
 * paragraph.  For PDFs and EPUBs, an odd number of underscores in a line
 * after wrapping is harmless.  The PDF underliner removes the underscores.
 * The EPUB underliner will put the open and close tags on different lines,
 * which is good XHTML.

 * In each case, the user will notice the incorrect output.
 */

int rtf_escape_line( struct string *new, char *line, int header )
{
   int len, underline = 0;
   char *ptr, *last = NULL;
   struct string *utf8 = NULL;

   underline = 0;

   for( ptr = line; *ptr; ++ptr )
      switch( *ptr )
      {
         case '{':
            if ( string_concat( new, "\\'7B" ))
               goto ERR;
            break;

         case '}':
            if ( string_concat( new, "\\'7D" ))
               goto ERR;
            break;

         case '\\':
            if ( utf8 == NULL && ( utf8 = string_make( NULL )) == NULL )
               goto ERR;

            string_clear( utf8 );
            len = process_accent(( unsigned char *)ptr + 1, utf8 );

            if ( len <= 0 )
            {
               if ( string_concat( new, "\\'5C" ))
                  goto ERR;
            }
            else
            {
               char buffer[ 8 ];
               unsigned int val;
               int code;

               ptr += len;

               if ( ! ( val = utf8_to_scalar(( unsigned char *)utf8->str )))
                  goto ERR;

               if ( val > 65535 )
                  code = 0;
               else if ( val > 32767 )
                  code = ( int )val - 65536;
               else
                  code = val;

               snprintf( buffer, sizeof( buffer ), "\\u%d?", code );

               if ( string_concat( new, buffer ))
                  return 1;
            }
            break;

         case ';':
            if ( header )
            {
               if ( last != ptr - 1 && string_concat( new, ", " ))
                  goto ERR;

               last = ptr;
            }
            else if ( string_append( new, *ptr ))
               goto ERR;

            break;

         case '_':
            if ( string_concat( new, ( underline ? "}" : "{\\ul " )))
               goto ERR;

            underline ^= 1;
            break;

         default:
            if (( unsigned char )*ptr < 128 )
            {
               if ( string_append( new, *ptr ))
                  goto ERR;
            }
            else
            {
               char buffer[ 8 ];
               unsigned int val;
               int code;

               if ( ! ( val = utf8_to_scalar(( unsigned char *)ptr )))
                  goto ERR;

               /*
                * Unicode escapes are signed 16-bit decimal numbers.  Code
                * points above 32767 must be negative numbers.  Code points
                * above the BMP are expressed as surrogate pairs.  We don't
                * support that.  Charles only knows about the Latin
                * ranges of Unicode.
                */

               if ( val > 65535 )
                  code = 0;
               else if ( val > 32767 )
                  code = ( int )val - 65536;
               else
                  code = val;

               snprintf( buffer, sizeof( buffer ), "\\u%d?", code );

               if ( string_concat( new, buffer ))
                  return 1;

               for( ++ptr; ( *ptr & 0xC0 ) == 0x80; )
                  ++ptr;

               --ptr;
            }
      }

   if ( utf8 != NULL )
      string_free( utf8 );

   if ( underline && string_append( new, '}' ))
      return 1;

   return 0;

ERR:
   if ( utf8 != NULL )
      string_free( utf8 );

   return 1;
}

int rtf_escape_lines( struct stack *lines, int header )
{
   struct string *new = NULL;
   int n;

   if (( new = string_make( NULL )) == NULL )
      return 1;

   for( n = 0; n < lines->used; ++n )
   {
      char *ptr;

      if ( rtf_escape_line( new, lines->values[ n ].ptr, header ))
         goto ERR;

      if (( ptr = str_dup( new->str, new->used )) == NULL )
         goto ERR;

      free( lines->values[ n ].ptr );
      lines->values[ n ].ptr = ptr;
      string_clear( new );
   }

   string_free( new );
   return 0;

ERR:
   string_free( new );
   return 1;
}

struct stack *make_header_lines( struct meta *meta )
{
   struct stack *lines;

   if (( lines = split_and_normalize( meta->header )) == NULL && ( lines = stack_make()) == NULL )
      return NULL;

   switch( lines->used )
   {
      case 0:
         if ( add_header_line( lines, "Anonymous" ) || add_header_line( lines, "UNTITLED" ))
            goto ERR;
         break;

      case 1:
         if ( add_header_line( lines, "UNTITLED" ))
            goto ERR;
         break;
   }

   return lines;

ERR:
   free_items( lines );
   return NULL;
}

char *make_info_line( char *input )
{
   char *line;
   struct string *new;

   if ( input == NULL || ! *input )
      return str_dup( "UNKNOWN", 1 );

   if (( new = string_make( NULL )) == NULL )
      return NULL;

   if ( rtf_escape_line( new, input, 1 ))
   {
      string_free( new );
      return NULL;
   }

   if (( line = str_dup( new->str, new->used )) == NULL )
   {
      string_free( new );
      return NULL;
   }

   string_free( new );
   return line;
}

int print_section( struct para *para )
{
   struct string *new;
   float offset, vert;
   union stack_val val;
	int first = 1;

   if (( new = string_make( NULL )) == NULL )
      return 1;

   vert = ( LINES - ( para->lines->used * 2 - 1 )) / 2 * LEADING;

   while( para->lines->used )
   {
      offset = center_line( para->lines->values[ 0 ].ptr, 1 );

      printf( "{\\pard \\pvpg\\phpg \\posx%d\\posy%d %s\\plain\\qc\\sl480\\slmulti1\\f0\\b ",
               ( int)( offset * 7.2 * 20 ), ( int)( vert * 24 ), ( first ? "\\pagebb" : "" ));

      first = 0;
      
      if ( stack_shift( para->lines, &val ))
         goto ERR;

      if ( rtf_escape_line( new, val.ptr, 0 ))
      {
         free( val.ptr );
         goto ERR;
      }

      printf( "%s\\par}\n", new->str );
      string_clear( new );

      free( val.ptr );
      vert += LEADING;
   }

   string_free( new );
   return 0;

ERR:
   string_free( new );
   return 1;
}

int add_centered_rtf_lines( char *source, float vert, int title )
{
   struct stack *lines = NULL;
   struct string *new = NULL;
   union stack_val val;
   float offset;

   if (( new = string_make( NULL )) == NULL )
      goto ERR;

   if (( lines = split_and_normalize( source )) == NULL )
      goto ERR;

   while( lines->used )
   {
      offset = center_line( lines->values[ 0 ].ptr, 1 );

      printf( "{\\pard \\pvpg\\phpg \\posx%d\\posy%d \\plain\\qc\\sl480\\slmulti1\\f0%s",
               ( int)( offset * 7.2 * 20 ), ( int)( vert * 20 ),
               ( title ? "\\b " : " " ));

      if ( stack_shift( lines, &val ))
         goto ERR;

      if ( rtf_escape_line( new, val.ptr, 1 ))
      {
         free( val.ptr );
         goto ERR;
      }

      free( val.ptr );
      printf( "%s\\par}\n", new->str );
      string_clear( new );

      vert += LEADING;
   }

   string_free( new );
   free_items( lines );
   return 0;

ERR:
   if ( new != NULL )
      string_free( new );

   if ( lines != NULL )
      free_items( lines );

   return 1;
}

int add_top_rtf_lines( char *source, struct meta *meta, int left )
{
   struct stack *lines = NULL;
   struct string *new = NULL;
   char *computed, *ptr;
   union stack_val val;
   float vert, offset;
   int skip = 0;

   if (( new = string_make( NULL )) == NULL )
      goto ERR;

   if (( lines = split_and_normalize( source )) == NULL )
      goto ERR;

   vert = RTF_TOP;

   while( lines->used )
   {
      if ( ! left )
      {
         ptr = ( char *)lines->values[ 0 ].ptr;

         if ( *ptr && *ptr == '$' && ( ptr[ 1 ] == 'D' || ptr[ 1 ] == 'W' || ptr[ 1 ] == 'P' ))
         {
            if ( ptr [ 1 ] == 'P' )
            {
               ++skip;
               goto SKIP;
            }

            if ( ptr[ 1 ] == 'D' )
            {
               if (( computed = make_date()) == NULL )
                  goto ERR;
            }
            else
            {
               if (( computed = make_words( meta->words, &ptr[ 2 ] )) == NULL )
                  goto ERR;
            }

            offset = 7.2 * right_justify( computed );

            if ( rtf_escape_line( new, computed, 1 ))
            {
               free( computed );
               goto ERR;
            }

            free( computed );

            printf( "{\\pard \\pvpg\\phpg \\posx%d\\posy%d \\plain\\qr\\sl480\\slmulti1\\f0 %s\\par}\n",
                    ( int )( offset * 20 ), ( int)( vert * 20 ), new->str );

            string_clear( new );

            if ( stack_shift( lines, &val ))
               goto ERR;

            free( val.ptr );
            vert += LEADING;
            continue;
         }
      }

      offset = ( left ? RTF_LEFT : 7.2 * right_justify( lines->values[ 0 ].ptr ));

      printf( "{\\pard \\pvpg\\phpg \\posx%d\\posy%d \\plain%s\\sl480\\slmulti1\\f0 ",
              ( int)( offset * 20 ), ( int )( vert * 20 ),
              ( left ? "\\ql" : "\\qr" ));

SKIP:
      if ( stack_shift( lines, &val ))
         goto ERR;

      if ( skip )
      {
         skip = 0;
         free( val.ptr );
         continue;
      }

      if ( rtf_escape_line( new, ( char *)val.ptr, 1 ))
      {
         free( val.ptr );
         goto ERR;
      }

      free( val.ptr );
      printf( "%s \\par}\n", new->str );
      string_clear( new );

      vert += LEADING;
   }

   string_free( new );
   free_items( lines );
   return 0;

ERR:
   if ( new != NULL )
      string_free( new );

   if ( lines != NULL )
      free_items( lines );

   return 1;
}

int add_rtf_title_page( struct meta *meta )
{
   char *strings[] = { "UNTITLED", "Anonymous", "Today", "me@nowhere.net" };

   if ( add_centered_rtf_lines(( meta->title == NULL ? strings[ 0 ] : meta->title ), RTF_HALF - LEADING * 5, 1 ))
      goto ERR;

   if ( add_centered_rtf_lines(( meta->author == NULL ? strings[ 1 ] : meta->author ), RTF_HALF + LEADING * 2, 0 ))
      goto ERR;

   if ( add_top_rtf_lines(( meta->revision == NULL ? strings[ 2 ] : meta->revision ), meta, 0 ))
      goto ERR;

   if ( add_top_rtf_lines(( meta->contact == NULL ? strings[ 3 ] : meta->contact ), meta, 1 ))
      goto ERR;

   return 0;

ERR:
   return 1;
}

void print_rtf( int newpage )
{
   int section = 0;

   struct tsml *tree;
   struct para *paras, *ptr;
   struct meta *meta;
   struct stack *hlines;

   char *title, *author,
      *body = "\\ql\\fi720\\sl480\\slmulti1\\f0",
      *body_break = "\\ql\\fi720\\sl480\\pagebb\\slmulti1\\f0",
      *pause = "\\qc\\sl480\\slmulti1\\f0",
      *pause_break = "\\qc\\sl480\\pagebb\\slmulti1\\f0",
      *chapter_break = "\\qc\\sl480\\slmulti1\\pagebb\\keepn\\f0\\b",
      *chapter_break_halfpage = "\\qc\\sl480\\slmulti1\\pagebb\\sb5320\\keepn\\f0\\b",
      *chapter_space = "\\qc\\sl480\\slmulti1\\sb532\\keepn\\f0\\b";

   if (( tree = make_tsml_tree( stdin )) == NULL )
      return;

   if (( meta = make_meta()) == NULL )
   {
      free_tsml_tree( tree );
      return;
   }

   if (( paras = make_para_list( tree, meta, 0 )) == NULL )
   {
      free_meta( meta );
      free_tsml_tree( tree );
      return;
   }

   free_tsml_tree( tree );

   if (( hlines = make_header_lines( meta )) == NULL )
   {
      free_meta( meta );
      free_para_list( paras );
      return;
   }

   if ( rtf_escape_lines( hlines, 0 ))
   {
      free_meta( meta );
      free_para_list( paras );
      free_items( hlines );
      return;
   }

   if (( title = make_info_line( meta->title )) == NULL )
   {
      free_meta( meta );
      free_para_list( paras );
      free_items( hlines );
      return;
   }

   if (( author = make_info_line( meta->author )) == NULL )
   {
      free( title );
      free_meta( meta );
      free_para_list( paras );
      free_items( hlines );
      return;
   }

   /*
    * Do not remove \plain\f0 from \header or the empty paragraph.
    * These are workarounds for bugs in MS Word.
    */

   printf( "{\\rtf1\\ansi\\deff0 {\\fonttbl {\\f0\\fmodern Courier;}} \\fs24\n"
           "\\margt1440\\margl1440\\margr1440\\margb1440\\widowctrl\n"
           "{\\stylesheet {\\s0 %s Normal;}\n"
           "{\\s1 %s Normal New Page;}\n"
           "{\\s2 %s Heading 1 No Space;}\n"
           "{\\s3 %s Heading 1 New Page;}\n"
           "{\\s4 %s Heading 1 ;}\n"
           "{\\s5 %s Pause;}\n"
           "{\\s6 %s Pause New Page;}}\n"
           "{\\info {\\title %s} {\\author %s}}\n"
           "{\\header \\pard\\qr\\plain\\f0 %s / %s / \\chpgn\\par}\n"
           "\\titlepg {\\headerf \\pard\\qr\\plain\\f0 \\'20 \\par}\n"
           "{\\pard\\plain\\par}\n",

           body, body_break,
           chapter_break, chapter_break_halfpage, chapter_space,
           pause, pause_break,
           title, author,
           ( char *)hlines->values[ 0 ].ptr, ( char *)hlines->values[ 1 ].ptr );

   free( title );
   free( author );
   free_items( hlines );

   if ( add_rtf_title_page( meta ))
   {
      free_meta( meta );
      free_para_list( paras );
      return;
   }

   free_meta( meta );

   for( ptr = paras; ptr != NULL; ptr = ptr->next )
   {
      int n;

      if ( rtf_escape_lines( ptr->lines, 0 ))
         break;

      switch( ptr->type )
      {
         case BODY:
            printf( "{\\pard \\s%d %s\n",
                    (( ptr == paras || section ) ? 1 : 0 ),
                    (( ptr == paras || section ) ? body_break : body ));
            break;

         case EMPHASIZED:
            printf( "{\\pard \\s%d \\i %s\n",
                    (( ptr == paras || section ) ? 1 : 0 ),
                    (( ptr == paras || section ) ? body_break : body ));
            break;

         case INTERRUPT:
            printf( "{\\pard \\s%d %s\n",
                    (( ptr == paras || section ) ? 6 : 5 ),
                    (( ptr == paras || section ) ? pause_break : pause ));
            break;

         case PART:
         case SECTION:
            print_section( ptr );
            ++section;
            continue;

         case CHAPTER:
            printf( "{\\pard \\s%d %s\n",
                    ( newpage ? 3 : (( ptr == paras || section ) ? 2 : 4 )),
                    ( newpage ? chapter_break_halfpage : (( ptr == paras || section ) ? chapter_break : chapter_space )));
      }

      for( n = 0; n < ptr->lines->used; ++n )
         printf( "%s \n", ( char *)ptr->lines->values[ n ].ptr );

      fputs( "\\par}\n", stdout );
      section = 0;
   }

   free_para_list( paras );
   fputc( '}', stdout );
}
