/*
 * Deflater Copyright (c) 2018, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "zlib.h"

void redirect_stdin( char *infile )
{
   FILE *file;

   if ( infile == NULL || ( *infile == '-' && ! infile[ 1 ] ))
      return;

   if (( file = fopen( infile, "r" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"r\" ): %s", infile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdin );
   stdin = file;
}

void redirect_stdout( char *outfile )
{
   FILE *out;

   if ( outfile == NULL )
      return;

   unlink( outfile );
   if (( out = fopen( outfile, "w" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"w\" ): %s\n", outfile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdout );
   stdout = out;
}

int deflater( int *slen )
{
   int len, flush, total;
   z_stream stream;
   unsigned char in_buffer[ 131072 ], out_buffer[ 131072 ];

   stream.zalloc = Z_NULL;
   stream.zfree  = Z_NULL;
   stream.opaque = Z_NULL;
   stream.avail_in = 0;
   stream.next_in = Z_NULL;

   if ( deflateInit( &stream, 9 ) != Z_OK )
      exit( 1 );

   *slen = total = 0;
   
   while(( len = fread( in_buffer, 1, sizeof( in_buffer ), stdin )))
   {
      total += len;

      stream.avail_in = len;
      stream.next_in = in_buffer;

      flush = ( feof( stdin ) ? Z_FINISH : Z_NO_FLUSH );

      do
      {
         stream.avail_out = sizeof( out_buffer );
         stream.next_out = out_buffer;

         if ( deflate( &stream, flush ) == Z_STREAM_ERROR )
            exit( 1 );

         len = sizeof( out_buffer ) - stream.avail_out;
         *slen += len;
         
         if ( ! fwrite( out_buffer, 1, len, stdout ))
            exit( 1 );
      }
      while( ! stream.avail_out );
   }

   deflateEnd( &stream );
   return total;
}

int main( int argc, char **argv )
{
   int stream, total;

   if ( argc > 1 )
      redirect_stdin( argv[ 1 ] );

   if ( argc > 2 )
      redirect_stdout( argv[ 2 ] );

   total = deflater( &stream );
   
   fprintf( stderr, "\"   /Length  %d\\n\"\n"
                    "\"   /Length1 %d >>\\n\"\n", stream, total );
   return 0;
}
