/*
 * LibIndex Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <syslog.h>
#include <stdarg.h>

union index_keyval
{
   void *ptr;
   float num;
};

struct index
{
   union index_keyval key, value;
   struct index *left, *right, *parent;
};

struct index_list
{
   union index_keyval key, value;
};

struct index_stack
{
   int free, used;
   struct index **top, **values;
};

struct index_stack *index_make_stack();
struct index *index_stack_push( struct index_stack *, struct index * );

struct index *index_init( int, struct index_list *,
                          int ( * )( const union index_keyval *,  const union index_keyval * ));

struct index *index_lookup( const struct index *, int ( * )( const union index_keyval, const union index_keyval ),
                            const union index_keyval );

struct index *index_first( const struct index * );
struct index *index_last( const struct index * );

struct index *index_next( const struct index * );
struct index *index_previous( const struct index * );

struct index *index_traverse( const struct index *, void ( * )( struct index *, void * ), void *, int );
int index_destroy( struct index *, void ( * )( union index_keyval, union index_keyval ));

#define INDEX_STACK_PUSH( _s_, _o_ ) ( ! ( _s_ )->free ? index_stack_push( _s_, _o_ ) : \
   ( ! ( _s_ )->used ? ( --( _s_ )->free, ++( _s_ )->used, *( _s_ )->top = _o_ ) : \
     ( ++( _s_ )->top, --( _s_ )->free, ++( _s_ )->used, *( _s_ )->top = _o_ )))

#define INDEX_STACK_POP( _s_ ) ( !( _s_ )->used ? NULL : ( ++( _s_ )->free, --( _s_ )->used, \
   ((( _s_ )->top == ( _s_ )->values ) ? *( _s_ )->top : ( --( _s_ )->top, *(( _s_ )->top + 1 )))))

#define INDEX_STACK_CLEAR( _s_ ) ( _s_ )->free += ( _s_ )->used; ( _s_)->used = 0; ( _s_ )->top = ( _s_ )->values
#define INDEX_STACK_FREE( _s_ ) free(( _s_ )->values ); free(( _s_ ))

struct index *index_lookup( const struct index *tree,
                            int ( *comparator )( const union index_keyval, const union index_keyval ),
                            const union index_keyval key )
{
   struct index *result = NULL;
   int order;

   for( ; tree != NULL; )

      if (( order = comparator( key, tree->key )) < 0 )
         tree = tree->left;
      else if ( ! order )
      {
         result = ( struct index *)tree;
         tree = NULL;
      }
      else
         tree = tree->right;

   return result;
}

struct index *index_previous( const struct index *tree )
{
   const struct index *node;

   if ( tree == NULL )
      return NULL;

   for( node = tree->left; node != NULL; node = node->right )
      if ( node->right == NULL )
         return ( struct index *)node;

   node = tree;

   if ( node->parent == NULL )
      return NULL;

   if ( node->parent->right == node )
      return node->parent;

   for( node = node->parent; node != NULL; node = node->parent )
      if ( node->parent != NULL && node->parent->right == node )
         return node->parent;

   return NULL;
}

struct index *index_next( const struct index *tree )
{
   const struct index *node;

   if ( tree == NULL )
      return NULL;

   for( node = tree->right; node != NULL; node = node->left )
      if ( node->left == NULL )
         return ( struct index *)node;

   node = tree;

   if ( node->parent == NULL )
      return NULL;

   if ( node->parent->left == node )
      return node->parent;

   for( node = node->parent; node != NULL; node = node->parent )
      if ( node->parent != NULL && node->parent->left == node )
         return node->parent;

   return NULL;
}

struct index *index_first( const struct index *tree )
{
   while( tree != NULL && tree->left != NULL )
      tree = tree->left;

   return ( struct index *)tree;
}

struct index *index_last( const struct index *tree )
{
   while( tree != NULL && tree->right != NULL )
      tree = tree->right;

   return ( struct index *)tree;
}

struct index *index_traverse( const struct index *tree, void ( *func )( struct index *, void * ),
                              void *data, int order )
{
   struct index *node;
   struct index_stack *stack;

   if ( tree == NULL )
      return NULL;

   if (( stack = index_make_stack()) == NULL )
      return NULL;

   for( node = ( struct index *)tree; ; )
   {
      for( ; node != NULL; node = ( order < 0 ? node->right : node->left ))
         if ( INDEX_STACK_PUSH( stack, node ) == NULL )
            return NULL;

      if (( node = INDEX_STACK_POP( stack )) == NULL )
         break;

      func( node, data );
      node = ( order < 0 ? node->left : node->right );
   }

   INDEX_STACK_FREE( stack );
   return ( struct index *)tree;
}

int index_destroy( struct index *tree, void ( *func )( union index_keyval, union index_keyval ))
{
   struct index *node;
   struct index_stack *stack;

   if ( tree == NULL )
      return 0;

   if (( stack = index_make_stack()) == NULL )
      return 1;

   for( ; ; )
   {
      for( node = tree; node != NULL; node = node->left )
         if ( INDEX_STACK_PUSH( stack, node ) == NULL )
            return 1;

      if (( node = INDEX_STACK_POP( stack )) == NULL )
         break;

      if ( func != NULL )
         func( node->key, node->value );

      tree = node->right;
      free( node );
   }

   INDEX_STACK_FREE( stack );
   return 0;
}

struct index *index_make_tree( struct index_list *list, struct index *parent, int start, int end )
{
   struct index *node;
   int midway;

   if ( start < 0 || start > end )
      return NULL;

   if (( node = malloc( sizeof( struct index ))) == NULL )
      return NULL;

   midway = ( start + end ) / 2;

   node->key = list[ midway ].key;
   node->value = list[ midway ].value;
   node->parent = parent;

   node->left = index_make_tree( list, node, start, midway - 1 );
   node->right = index_make_tree( list, node, midway + 1, end );

   return node;
}

struct index *index_init( int nkeys, struct index_list *list,
                          int ( *comparator )( const union index_keyval *,  const union index_keyval * ))
{
   qsort( list, nkeys, sizeof( struct index_list ), ( int ( * )( const void *, const void *))comparator );
   return index_make_tree( list, NULL, 0, nkeys - 1 );
}

struct index *index_stack_push( struct index_stack *a, struct index *o )
{
   if ( a->free == 0 )
   {
      a->values = realloc( a->values, sizeof( void * ) * ( a->used + 64 ));

      if ( a->values == NULL )
      {
         syslog( LOG_ERR, "realloc: %m.\n" );
         return NULL;
      }

      a->free = 64;
      a->top = &a->values[ a->used - 1 ];
   }

   if ( a->used )
      ++a->top;

   *a->top = o;
   --a->free;
   ++a->used;

   return o;
}

struct index_stack *index_make_stack()
{
   struct index_stack *a;

   if (( a = malloc( sizeof( struct index_stack ))) == NULL )
   {
      syslog( LOG_ERR, "malloc(): %m" );
      return NULL;
   }

   if (( a->values = malloc( sizeof( void * ) * 64 )) == NULL )
   {
      syslog( LOG_ERR, "malloc(): %m" );
      free( a );
      return NULL;
   }

   a->free = 64;
   a->used = 0;
   a->top = a->values;

   return a;
}
