/*
 * LibIndex Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>

#include "index.h"

#define MAX_NODES 1024

int number_comparator( const union index_keyval first, const union index_keyval second )
{
   return first.num - second.num;
}

int init_number_comparator( const union index_keyval *first, const union index_keyval *second )
{
   return first->num - second->num;
}

int init_string_comparator( const union index_keyval *first, const union index_keyval *second )
{
   char *ptr1, *ptr2;

   ptr1 = ( char *)first->ptr;
   ptr2 = ( char *)second->ptr;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      return 0;

   if ( ! *ptr1 )
      return -1;

   if ( ! *ptr2 )
      return 1;

   return ( *ptr1 - *ptr2 );
}

int string_comparator( const union index_keyval first, const union index_keyval second )
{
   char *ptr1, *ptr2;

   ptr1 = ( char *)first.ptr;
   ptr2 = ( char *)second.ptr;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      return 0;

   if ( ! *ptr1 )
      return -1;

   if ( ! *ptr2 )
      return 1;

   return ( *ptr1 - *ptr2 );
}

void printer( struct index *node, void *data )
{
   printf( "%s -> %s\n", ( char *)node->key.ptr, ( char *)node->value.ptr );
}

void parents( struct index *node, void *data )
{
   printf( "%s -> %s\n", ( char *)node->key.ptr, ( node->parent == NULL ? "NULL" : ( char *)node->parent->key.ptr ));

   if ( data != NULL )
      INDEX_STACK_PUSH(( struct index_stack *)data, node );
}

void progress( struct index *node, void *data )
{
   printf( "%f ", node->key.num );
}

void string_keys()
{
   struct index_list index[ 10 ];
   struct index_stack *stack;
   struct index *tree = NULL, *node, **ptr;
   int n;

   index[ 0 ].key.ptr = "Z";
   index[ 1 ].key.ptr = "X";
   index[ 2 ].key.ptr = "Y";
   index[ 3 ].key.ptr = "W";
   index[ 4 ].key.ptr = "V";
   index[ 5 ].key.ptr = "U";
   index[ 6 ].key.ptr = "T";
   index[ 7 ].key.ptr = "S";
   index[ 8 ].key.ptr = "R";
   index[ 9 ].key.ptr = "Q";

   index[ 0 ].value.ptr = "Z";
   index[ 1 ].value.ptr = "X";
   index[ 2 ].value.ptr = "Y";
   index[ 3 ].value.ptr = "W";
   index[ 4 ].value.ptr = "V";
   index[ 5 ].value.ptr = "U";
   index[ 6 ].value.ptr = "T";
   index[ 7 ].value.ptr = "S";
   index[ 8 ].value.ptr = "R";
   index[ 9 ].value.ptr = "Q";

   puts( "index_init( 10 string keys )..." );
   tree = index_init( 10, index, init_string_comparator );

   puts( "index_traverse( 1 )..." );
   index_traverse( tree, printer, NULL, 1 );

   puts( "index_traverse( -1 )..." );
   index_traverse( tree, printer, NULL, -1 );

   puts( "index_lookup() keys X, Y, and Z..." );
   index[ 0 ].key.ptr = "X";
   node = index_lookup( tree, string_comparator, index[ 0 ].key );
   printer( node, NULL );

   index[ 0 ].key.ptr = "Y";
   node = index_lookup( tree, string_comparator, index[ 0 ].key );
   printer( node, NULL );

   index[ 0 ].key.ptr = "Z";
   node = index_lookup( tree, string_comparator, index[ 0 ].key );
   printer( node, NULL );

   puts( "parent pointer test:" );
   stack = index_make_stack();
   index_traverse( tree, parents, stack, 1 );

   puts( "stack test:" );
   for( n = 0, ptr = stack->values; n < stack->used; ++n, ++ptr )
      puts(( *ptr )->key.ptr );

   INDEX_STACK_FREE( stack );

   puts( "successor/predecessor checks:" );
   node = index_first( tree );
   printf( "index_first(): %s\n", ( char *)node->key.ptr );

   while(( node = index_next( node )) != NULL )
      printf( "index_next(): %s\n", ( char *)node->key.ptr );

   node = index_last( tree );
   printf( "index_last(): %s\n", ( char *)node->key.ptr );

   while(( node = index_previous( node )) != NULL )
      printf( "index_previous(): %s\n", ( char *)node->key.ptr );

   puts( "index_destroy()..." );
   index_destroy( tree, NULL );
}

void numerical_keys()
{
   struct index *tree = NULL;
   struct index_list index[ MAX_NODES ];
   int n;

   /*
    * This is a contrived example.  If you had contiguous numerical
    * keys, you would, of course, use an array.
    */

   printf( "index_init( %d number keys )...\n", MAX_NODES );

   for( n = 0; n < MAX_NODES; ++n )
      index[ n ].key.num = index[ n ].value.num = n;

   tree = index_init( MAX_NODES, index, init_number_comparator );

   puts( "index_traverse( 1 )..." );
   index_traverse( tree, progress, NULL, 1 );

   puts( "\nindex_destroy()..." );
   index_destroy( tree, NULL );
}

int main( int argc, char **argv )
{
   string_keys();
   numerical_keys();
   return 0;
}
