/*
 * StringStack Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>

#define STACK_INC 64
#define STRING_INC 128

union stack_val
{
   float num;
   void *ptr;
};

struct stack
{
   int free, used;
   union stack_val *top, *values;
};

struct string
{
   int free, used;
   char *top, *str;
};

struct stack *stack_make();
void stack_free( struct stack * );
void stack_clear( struct stack * );

int stack_push( struct stack *, union stack_val );
union stack_val *stack_pop( struct stack * );

int stack_shift( struct stack *, union stack_val * );
int stack_unshift( struct stack *, union stack_val );

struct string *string_make( char * );
void string_free( struct string * );

int string_append( struct string *, char );
int string_prepend( struct string *, char );

int string_concat( struct string *, char * );
int string_precat( struct string *, char * );
int string_assign( struct string *, char * );

int string_merge( struct string *, struct string * );
void string_clear( struct string * );

struct stack *stack_make()
{
   struct stack *a;

   if (( a = malloc( sizeof( struct stack ))) == NULL )
      return NULL;

   if (( a->values = malloc( sizeof( union stack_val ) * STACK_INC )) == NULL )
   {
      free( a );
      return NULL;
   }

   a->free = STACK_INC;
   a->used = 0;
   a->top = a->values;

   return a;
}

void stack_free( struct stack *a )
{
   free( a->values );
   free( a );
}

void stack_clear( struct stack *s )
{
   s->free += s->used;
   s->used = 0;
   s->top = s->values;
}

int stack_unshift( struct stack *a, union stack_val o )
{
   union stack_val *vptr;
   int n;

   if ( a->free == 0 )
   {
      if (( a->values = realloc( a->values, sizeof( union stack_val ) * ( a->used + STACK_INC ))) == NULL )
         return 1;

      a->free = STACK_INC;
      a->top = &a->values[ a->used - 1 ];
   }

   for( vptr = &a->values[ a->used - 1 ], n = 0; n < a->used; ++n, --vptr )
      *( vptr + 1 ) = *vptr;

   a->values[ 0 ] = o;

   if ( a->used )
      ++a->top;

   ++a->used;
   --a->free;

   return 0;
}

int stack_push( struct stack *a, union stack_val o )
{
   if ( a->free == 0 )
   {
      if (( a->values = realloc( a->values, sizeof( union stack_val ) * ( a->used + STACK_INC ))) == NULL )
         return 1;

      a->free = STACK_INC;
      a->top = &a->values[ a->used - 1 ];
   }

   if ( a->used )
      ++a->top;

   *a->top = o;
   --a->free;
   ++a->used;

   return 0;
}

int stack_shift( struct stack *stack, union stack_val *result )
{
   union stack_val *vptr;
   int n;

   if ( ! stack->used )
      return 1;

   *result = *stack->values;

   --stack->used;
   ++stack->free;

   if ( stack->top != stack->values )
      --stack->top;

   for( vptr = stack->values, n = 0; n < stack->used; ++n, ++vptr )
      *vptr = *( vptr + 1 );

   return 0;
}

union stack_val *stack_pop( struct stack *stack )
{
   union stack_val *result = NULL;

   if ( ! stack->used )
      return result;

   result = stack->top;

   ++stack->free;
   --stack->used;

   if ( stack->top != stack->values )
      --stack->top;

   return result;
}

struct string *string_make( char *o )
{
   struct string *s;

   if (( s = malloc( sizeof( struct string ))) == NULL )
      return NULL;

   if (( s->str = malloc( STRING_INC + 1 )) == NULL )
   {
      free( s );
      return NULL;
   }

   *s->str = '\0';
   s->free = STRING_INC;
   s->used = 0;
   s->top = s->str;

   if ( o != NULL && string_concat( s, o ))
   {
      string_free( s );
      return NULL;
   }

   return s;
}

void string_free( struct string *s )
{
   free( s->str );
   free( s );
}

int string_append( struct string *s, char c )
{
   if ( s->free == 0 )
   {
      if (( s->str = realloc( s->str, s->used + 1 + STRING_INC )) == NULL )
         return 1;

      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
   }

   ++s->used;
   --s->free;
   *s->top++ = c;
   *s->top = '\0';

   return 0;
}

int string_prepend( struct string *s, char c )
{
   char *ptr1, *ptr2;

   if ( s->used == 0 )
   {
      string_append( s, c );
      return 0;
   }

   if ( s->free == 0 )
   {
      if (( s->str = realloc( s->str, s->used + 1 + STRING_INC )) == NULL )
         return 1;

      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
   }

   ptr1 = &s->str[ s->used ];
   ptr2 = &s->str[ s->used + 1 ];

   while( ptr1 >= s->str )
      *ptr2-- = *ptr1--;

   s->str[ 0 ] = c;

   ++s->used;
   ++s->top;
   --s->free;

   return 0;
}

int string_concat( struct string *s, char *c )
{
   while( *c )
      if ( ! s->free )
      {
         if ( string_append( s, *c++ ))
            return 1;
      }
      else
      {
         ++s->used;
         --s->free;
         *s->top++ = *c++;
         *s->top = '\0';
      }

   return 0;
}

int string_assign( struct string *s, char *c )
{
   s->free += s->used;
   s->used = 0;
   s->top = s->str;
   *s->str = '\0';

   if ( string_concat( s, c ))
      return 1;

   return 0;
}

int string_precat( struct string *s, char *c )
{
   int len;
   char *old = NULL, *ptr1, *ptr2;

   for( len = 0, ptr1 = c; *ptr1; ++ptr1 )
      ++len;

   if ( len <= s->free )
   {
      ptr1 = s->top + len;
      ptr2 = s->top;

      while( ptr2 >= s->str )
         *ptr1-- = *ptr2--;

      s->used += len;
      s->free -= len;
      s->top = &s->str[ s->used ];
   }
   else
   {
      old = s->str;

      if (( s->str = malloc( s->used + len + 1 + STRING_INC )) == NULL )
         return 1;

      s->used = s->used + len;
      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
      *s->top = '\0';
   }

   ptr1 = s->str;
   ptr2 = c;

   while( *ptr2 )
      *ptr1++ = *ptr2++;

   if ( old != NULL )
   {
      ptr2 = old;

      while( *ptr2 )
         *ptr1++ = *ptr2++;

      free( old );
   }

   return 0;
}

int string_merge( struct string *a, struct string *b )
{
   if ( string_concat( a, b->str ))
      return 1;

   return 0;
}

void string_clear( struct string *s )
{
   s->free += s->used;
   s->used = 0;
   s->top = s->str;
   *s->top = '\0';
}

unsigned int string_UTF8_len( unsigned char *str )
{
   unsigned int n = 0;
   unsigned char *ptr;

   for( ptr = str; *ptr; ++ptr )
      if (( *ptr & 0xC0 ) != 0x80 )
         ++n;

   return n;
}

unsigned int string_UTF16_len( unsigned char *str )
{
   unsigned char *ptr;
   unsigned int n = 0, endian = 1;
   unsigned int code;

   ptr = str;

   if ( ptr[ 0 ] == 0xFF && ptr[ 1 ] == 0xFE )
   {
      endian = 0;
      ptr += 2;
   }

   if ( ptr[ 0 ] == 0xFE && ptr[ 1 ] == 0xFF )
      ptr += 2;

   while( *ptr )
   {
      if ( ! ptr[ 1 ] )
         break;

      if ( endian )
         code = ( ptr[ 0 ] << 8 ) + ptr[ 1 ];
      else
         code = ( ptr[ 1 ] << 8 ) + ptr[ 0 ];

      if ( code >= 0xD800 && code <= 0xDBFF )
      {
         if ( ! ptr[ 2 ] || ! ptr[ 3 ] )
            break;

         ptr += 2;
      }

      ptr += 2;
      ++n;
   }

   return n;
}

unsigned int string_UTF8_to_value( unsigned char *input )
{
   unsigned int value = 0;

   switch( *input & 0xF0 )
   {
      case 0xF0:
         if ( ! input[ 1 ] || ! input[ 2 ] || ! input[ 3 ] )
            goto ERR;

         value += (( input[ 0 ] & 0x07 ) << 18 ) + (( input[ 1 ] & 0x3F ) << 12 ) +
                  (( input[ 2 ] & 0x3F ) << 6 )  + ( input[ 3 ] & 0x3F );
         break;

      case 0xE0:
         if ( ! input[ 1 ] || ! input[ 2 ] )
            goto ERR;

         value += (( input[ 0 ] & 0x0F ) << 12 ) + (( input[ 1 ] & 0x3F ) << 6 ) +
                  ( input[ 2 ] & 0x3F );
         break;

      case 0xC0:
      case 0xD0:
         if ( ! input[ 1 ] )
            goto ERR;

         value += (( input[ 0 ] & 0x1F ) << 6 ) + ( input[ 1 ] & 0x3F );
         break;

      default:
         value = ( unsigned int )*input;
   }

ERR:
   return value;
}

struct string *string_UTF8_to_UTF32BE( struct string *input )
{
   struct string *output;
   unsigned char *ptr;
   unsigned int wide, upper, lower;
   unsigned char byte1, byte2, byte3, byte4;

   if (( output = string_make( NULL )) == NULL )
      return NULL;

   for( ptr = ( unsigned char *)input->str; *ptr; ++ptr )

      if (( *ptr & 0xC0 ) != 0x80 )
      {
         if ( ! ( wide = string_UTF8_to_value( ptr )))
            goto ERR;

         upper = wide / 0x10000;
         lower = wide % 0x10000;

         byte1 = upper / 0x100;
         byte2 = upper % 0x100;

         byte3 = lower / 0x100;
         byte4 = lower % 0x100;

         if ( string_append( output, byte1 ) || string_append( output, byte2 ) ||
              string_append( output, byte3 ) || string_append( output, byte4 ))
            goto ERR;
      }

   return output;

ERR:
   string_free( output );
   return NULL;
}

struct string *string_UTF8_to_UTF16BE( struct string *input )
{
   struct string *output;
   unsigned char *ptr;
   unsigned int wide, high, low;
   unsigned char upper, lower;

   if (( output = string_make( NULL )) == NULL )
      return NULL;

   if ( string_append( output, 0xFE ) || string_append( output, 0xFF ))
      goto ERR;

   for( ptr = ( unsigned char *)input->str; *ptr; ++ptr )

      if (( *ptr & 0xC0 ) != 0x80 )
      {
         if ( ! ( wide = string_UTF8_to_value( ptr )))
            goto ERR;

         /*
          * Surrogates mean bad utf-8.
          */

         if ( wide >= 0xD800 && wide <= 0xDFFF )
            goto ERR;

         if ( wide < 0x10000 )
         {
            upper = wide / 0x100;
            lower = wide % 0x100;

            if ( string_append( output, upper ) || string_append( output, lower ))
               goto ERR;
         }
         else
         {
            wide -= 0x10000;
            high = 0xD800 + (( wide & 0xFFC00 ) >> 10 );
            low  = 0xDC00 + ( wide & 0x003FF );

            upper = high / 0x100;
            lower = high % 0x100;

            if ( string_append( output, upper ) || string_append( output, lower ))
               goto ERR;

            upper = low / 0x100;
            lower = low % 0x100;

            if ( string_append( output, upper ) || string_append( output, lower ))
               goto ERR;
         }
      }

   return output;

ERR:
   string_free( output );
   return NULL;
}

struct string *string_UTF16_to_UTF8( struct string *input )
{
   struct string *output;
   unsigned char *ptr, endian = 1;
   unsigned int wide1, wide2, code;
   unsigned char byte1, byte2, byte3, byte4;

   if (( output = string_make( NULL )) == NULL )
      return NULL;

   ptr = ( unsigned char *)input->str;

   if ( ptr[ 0 ] == 0xFF && ptr[ 1 ] == 0xFE )
   {
      endian = 0;
      ptr += 2;
   }

   if ( ptr[ 0 ] == 0xFE && ptr[ 1 ] == 0xFF )
      ptr += 2;

   while( *ptr )
   {
      if ( ! ptr[ 1 ] )
         goto ERR;

      if ( endian )
         code = wide1 = ( ptr[ 0 ] << 8 ) + ptr[ 1 ];
      else
         code = wide1 = ( ptr[ 1 ] << 8 ) + ptr[ 0 ];

      wide2 = 0;

      if ( wide1 >= 0xD800 && wide1 <= 0xDBFF )
      {
         if ( ! ptr[ 2 ] || ! ptr[ 3 ] )
            goto ERR;

         if ( endian )
            wide2 = ( ptr[ 2 ] << 8 ) + ptr[ 3 ];
         else
            wide2 = ( ptr[ 3 ] << 8 ) + ptr[ 2 ];

         if ( wide2 < 0xDC00 || wide2 > 0xDFFF )
            goto ERR;

         code = (( wide1 - 0xD800 ) << 10 ) + ( wide2 - 0xDC00 ) + 0x10000;
         ptr += 2;
      }

      ptr += 2;

      if ( code < 0x80 )
      {
         if ( string_append( output, ( char )code ))
            goto ERR;
      }
      else if ( code < 0x800 )
      {
         byte1 = 0xC0 + (( code & 0x7C0 ) >> 6 );
         byte2 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ))
            goto ERR;
      }
      else if ( code < 0x10000 )
      {
         byte1 = 0xE0 + (( code & 0xF000 ) >> 12 );
         byte2 = 0x80 + (( code & 0xFC0 ) >> 6 );
         byte3 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ) || string_append( output, byte3 ))
            goto ERR;
      }
      else
      {
         byte1 = 0xF0 + (( code & 0x1C0000 ) >> 18 );
         byte2 = 0x80 + (( code & 0x3F000 ) >> 12 );
         byte3 = 0x80 + (( code & 0xFC0 ) >> 6 );
         byte4 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ) || string_append( output, byte3 ) ||
              string_append( output, byte4 ))
            goto ERR;
      }
   }

   return output;

ERR:
   string_free( output );
   return NULL;
}

struct string *string_UTF32BE_to_UTF8( struct string *input )
{
   struct string *output;
   unsigned char *ptr;
   unsigned int code;
   unsigned char byte1, byte2, byte3, byte4;

   if (( output = string_make( NULL )) == NULL )
      return NULL;

   for( ptr = ( unsigned char *)input->str; *ptr; ptr += 4 )
   {
      if ( ! ptr[ 1 ] || ! ptr[ 2 ] || ! ptr[ 3 ] )
         goto ERR;

      code = ( ptr[ 0 ] << 24 ) + ( ptr[ 1 ] << 16 ) + ( ptr[ 2 ] << 8 ) + ptr[ 3 ];

      /*
       * Surrogates mean bad UTF-32. Code points greater than Unicode last
       * code point mean bad UTF-32.
       */

      if (( code >= 0xD800 && code <= 0xDFFF ) || code > 0x10FFFF )
         goto ERR;

      if ( code < 0x80 )
      {
         if ( string_append( output, ( char )code ))
            goto ERR;
      }
      else if ( code < 0x800 )
      {
         byte1 = 0xC0 + (( code & 0x7C0 ) >> 6 );
         byte2 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ))
            goto ERR;
      }
      else if ( code < 0x10000 )
      {
         byte1 = 0xE0 + (( code & 0xF000 ) >> 12 );
         byte2 = 0x80 + (( code & 0xFC0 ) >> 6 );
         byte3 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ) || string_append( output, byte3 ))
            goto ERR;
      }
      else
      {
         byte1 = 0xF0 + (( code & 0x1C0000 ) >> 18 );
         byte2 = 0x80 + (( code & 0x3F000 ) >> 12 );
         byte3 = 0x80 + (( code & 0xFC0 ) >> 6 );
         byte4 = 0x80 + ( code & 0x3F );

         if ( string_append( output, byte1 ) || string_append( output, byte2 ) || string_append( output, byte3 ) ||
              string_append( output, byte4 ))
            goto ERR;
      }
   }

   return output;

ERR:
   string_free( output );
   return NULL;
}
