/*
 * Ephemera Copyright (c) 2014-2025, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Multiplexing blog server.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/sysctl.h>
#include <sys/un.h>
#include <sys/stat.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>

#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "data.h"
#include "sqlite.h"
#include "calendar.h"

#define PIDFILE "/var/run/ephemera.pid"

/*
 * Maximum length of configuration values.
 */

#define VARLEN 128

/*
 * Number of simultaneous connections.
 */

#define MAX_CONN 1024

/*
 * Number of events generated per connection.
 */

#define MAX_EVENT 3

/*
 * Max number of new connections accepted at one time
 */

#define NEWCONS 1024

/*
 * Maximum size of chunks of data written to client.  We multiplex by only
 * writing this much data each time we service a client connection.
 */

#define IOSIZE 32768

/*
 * KQueue Input and output queues.  We can add a maximum of 3 events to
 * the input queue for every item in the output queue, while processing
 * the output queue.
 */

struct kevent *inqueue = NULL;
struct kevent *outqueue = NULL;

extern char *optarg;

int in = 0, out = 0, fd = -1, logging = 0, testing = 0, qlen = MAX_CONN * MAX_EVENT,
    backlog = 1000, closed = 0, active = 0, in_memory = 1, max_conns = MAX_CONN;

volatile int terminate = 0, reload = 0;

char *interface = "", *port = "4000", *grp = "nobody", *user = "nobody",
     *unix_socket = NULL, *config_dir = "/usr/local/etc",
     *recent = NULL;

struct passwd *passwd;
struct group *group;

int p_size;

sqlite3 *db = NULL;
sqlite3_stmt *recent_sql = NULL, *oldest_sql = NULL, *one_sql = NULL,
   *search_sql = NULL, *era_sql = NULL, *count_sql = NULL;

char *title = NULL, *favicon = NULL, *meta_keywords, *blurb = NULL,
     *meta_language = NULL, *meta_robots = NULL, *meta_description = NULL,
     *path = NULL, *directory = NULL, *web_directory= NULL, *page_size = NULL;

char *var_names[] = { "title", "favicon", "meta_keywords", "blurb",
                      "meta_robots", "meta_description", "meta_language",
                      "path", "directory", "web_directory", "page_size", NULL };

char **var_ptrs[] = { &title, &favicon, &meta_keywords, &blurb,
                      &meta_robots, &meta_description, &meta_language,
                      &path, &directory, &web_directory, &page_size, NULL };

char desc[ 512 ], keys[ 512 ], lang[ 512 ], robs[ 512 ], fav[ 512 ], css[ 512 ], blb[ 512 ];

int permalink = 0;

struct stack *calendars = NULL;

void make_calendar_cache();

/*
 * Connection control block:
 */

struct ccb
{
   int sock;
   off_t written, count;
   struct stack *env, *params;
   struct string *response;
   char buffer[ 8 ], *header;
};

void close_on_exec( int fd )
{
   if ( fcntl( fd, F_SETFD, FD_CLOEXEC ) < 0 )
      syslog( LOG_ERR, "fcntl( F_SETFD, FD_CLOEXEC ): %m" );
}

void ev_set( int desc, short filter, u_short flags, struct ccb *item )
{
   struct kevent *kev;

   if ( in >= qlen )
      return;

   kev = &inqueue[ in++ ];

   kev->ident = desc;
   kev->filter = filter;
   kev->fflags = 0;
   kev->flags = flags;
   kev->udata = item;
}

void non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }
}

void set_options( int argc, char **argv )
{
   int i, specified = 0;

   while(( i = getopt( argc, argv, "xdm:f:l:p:i:u:g:")) != -1 )
      switch( i )
      {
         case 'd':
            in_memory = 0;
            break;

         case 'f':
            config_dir = optarg;
            break;

         case 'g':
            grp = optarg;
            break;

         case 'i':
            ++specified;
            interface = optarg;
            break;

         case 'l':
            unix_socket = optarg;
            break;

         case 'm':
            max_conns = strtol( optarg, NULL, 10 );
            break;

         case 'p':
            ++specified;
            port = optarg;
            break;

         case 'u':
            user = optarg;
            break;

         case 'x':
            testing = 1;
      }

   if ( max_conns <= 0 )
   {
      fprintf( stderr, "ephemera: max_conns <= 0: %d\n", max_conns );
      exit( 1 );
   }

   qlen = max_conns * MAX_EVENT;

   if (( passwd = getpwnam( user )) == NULL )
   {
      fprintf( stderr, "ephemera: user \"%s\" does not exist\n", user );
      exit( 1 );
   }

   if (( group = getgrnam( grp )) == NULL )
   {
      fprintf( stderr, "ephemera: group \"%s\" does not exist\n", grp );
      exit( 1 );
   }

   if ( specified && unix_socket != NULL )
   {
      fputs( "The -l option cannot be used together with the -i and -p options\n", stderr );
      exit( 1 );
   }
}

void become_daemon()
{
   int file;

   /*
    * Fork and let the parent die, continuing as child so we are not
    * a process group leader.  This is necessary for the call to setsid().
    */

   switch( fork() )
   {
      case -1:
         fprintf( stderr, "ephemera: fork(): %s\n", strerror( errno ));
         exit( 1 );

      case 0:
         break;

      default:
         exit( 0 );
   }

   fclose( stdout );
   fclose( stderr );
   fclose( stdin );

   stdin = fopen( "/dev/null", "r" );
   stdout = fopen( "/dev/null", "w" );
   stderr = fopen( "/dev/null", "w" );

   if ( stdin == NULL || stdout == NULL || stderr == NULL )
   {
      syslog( LOG_ERR, "fopen(): %m" );
      exit( 1 );
   }

   /*
    * Detach us from our controlling terminal, so job control and other
    * signals may not be sent to us from that terminal.
    */

   if ( setsid() < 0 )
   {
      syslog( LOG_ERR, "setsid(): %m" );
      exit( 1 );
   }

   /*
    * Write our pid to disk.
    */

   if (( file = open( PIDFILE, O_WRONLY | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP )) < 0 )
      syslog( LOG_WARNING, "open(): %m" );
   else
   {
      char buffer[ 16 ];

      snprintf( buffer, sizeof( buffer ), "%d", getpid() );
      write( file, buffer, strlen( buffer ));
      close( file );
   }

   /*
    * Make files inaccessible to the world.
    */

   umask( 7 );
}

void change_identity()
{
   if ( setgid( group->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "setgid(): %m" );
      exit( 1 );
   }

   if ( setuid( passwd->pw_uid ) < 0 )
   {
      syslog( LOG_ERR, "setuid(): %m" );
      exit( 1 );
   }
}

void start_listening_unix()
{
   struct sockaddr_un sa;

   if (( fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   unlink( unix_socket );
   bzero( &sa, sizeof( struct sockaddr_un ));
   strncpy( sa.sun_path, unix_socket, sizeof( sa.sun_path ) - 1 );  /* ensures NUL-terminated. */
   sa.sun_family = AF_UNIX;

   if ( bind( fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )))
   {
      syslog( LOG_ERR, "bind( %s ): %m", unix_socket );
      close( fd );
      exit( 1 );
   }

   if ( chown( unix_socket, passwd->pw_uid, group->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "chown( %s ): %m", unix_socket );
      close( fd );
      exit( 1 );
   }

   if ( chmod( unix_socket, S_IRWXU | S_IRWXG ) < 0 )
   {
      syslog( LOG_ERR, "chmod( %s, S_IRWXU | S_IRWXG ): %m", unix_socket );
      close( fd );
      exit( 1 );
   }

   if ( listen( fd, backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      close( fd );
      exit( 1 );
   }

   close_on_exec( fd );
   non_blocking( fd );
}

void start_listening()
{
   struct addrinfo hints, *res;
   int result;

   /*
    * This is the new, protocol-independent way of setting up a listening
    * socket.
    */

   bzero( &hints, sizeof( struct addrinfo ));
   hints.ai_flags = AI_PASSIVE;
   hints.ai_socktype = SOCK_STREAM;

   /*
    * If the user has not specified an interface, we listen on the
    * IPv6 wildcard address.
    */

   if (( result = getaddrinfo( ( *interface ? interface : NULL ), port, &hints, &res )))
   {
      syslog( LOG_ERR, "getaddrinfo(): %s", gai_strerror( result ));
      exit( 1 );
   }

   if ( res == NULL )
   {
      syslog( LOG_ERR, "getaddrinfo(): no interface found" );
      exit( 1 );
   }

   fd = socket( res->ai_family, res->ai_socktype, res->ai_protocol );

   if ( fd == -1 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   result = 1;

   /*
    * Allow duplicate bindings, so we can restart immediately.
    */

   if ( setsockopt( fd, SOL_SOCKET, SO_REUSEPORT, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_REUSEPORT ): %m" );

   /*
    * Try and detect connections which go idle for long periods of time.
    */

   if ( setsockopt( fd, SOL_SOCKET, SO_KEEPALIVE, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_KEEPALIVE ): %m" );

   /*
    * Make sure we can accept IPv4 traffic as well as IPv6, if we
    * are bound to the IPv6 wildcard address.  We are turning off
    * the option because result = 0.
    */

   result = 0;

   if ( ! *interface && setsockopt( fd, IPPROTO_IPV6, IPV6_BINDV6ONLY, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( IPV6_BINDV6ONLY ): %m" );

   if ( bind( fd, res->ai_addr, res->ai_addrlen ) < 0 )
   {
      syslog( LOG_ERR, "bind(): %m" );
      exit( 1 );
   }

   freeaddrinfo( res );

   if ( listen( fd, backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      exit( 1 );
   }

   close_on_exec( fd );
   non_blocking( fd );
}

void add_conn( int new )
{
   struct ccb *ptr;

   ptr = memory( sizeof( struct ccb ));
   close_on_exec( new );

   ptr->sock = new;
   ptr->header = NULL;

   ptr->response = make_string();

   ptr->env = make_stack();
   ptr->params = make_stack();

   ptr->count = ptr->written = 0;

   ev_set( ptr->sock, EVFILT_READ, EV_ADD, ptr );
   ev_set( ptr->sock, EVFILT_WRITE, EV_ADD | EV_DISABLE, ptr );
   ++active;
}

void remove_conn( struct ccb *item )
{
   char **ptr;

   closed = item->sock;
   close( item->sock );

   string_free( item->response );
   stack_free( item->env );

   if ( item->params->used )
      for( ptr = item->params->values; *ptr != NULL; ++ptr )
         free( *ptr );

   stack_free( item->params );

   if ( item->header != NULL )
      free( item->header );

   free( item );
   --active;
}

void accept_connection()
{
   int conn, i = 0;

   do
   {
      conn = accept( fd, NULL, NULL );

      if ( conn < 0 )
      {
         if ( errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
         {
            syslog( LOG_ERR, "accept(): %m" );
            exit( 1 );
         }

         return;
      }

      if ( active < max_conns )
         add_conn( conn );
      else
         close( conn );
   }
   while( ++i < NEWCONS );
}

char *find_next( char *ptr, int *len )
{
   while( --*len )
      if ( ! *ptr++ )
         break;

   if ( ! *len )
      return NULL;

   return ptr;
}

void init_env( char *header, int len, struct stack *env )
{
   char *ptr;
   int n;

   for( n = 0, ptr = header; ptr != NULL; ptr = find_next( ptr, &len ), ++n )
      STACK_PUSH( env, ptr )

   STACK_PUSH( env, NULL )

   return;
}

char *get_var( char *key, struct stack *stk )
{
   char **ptr, *ptr1, *ptr2;

   if ( stk->used )
      for( ptr = stk->values; *ptr != NULL; ptr += 2 )
      {
         ptr1 = *ptr;
         ptr2 = key;

         while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
         {
            ++ptr1;
            ++ptr2;
         }

         if ( ! *ptr1 && ! *ptr2 )
            return *( ++ptr );
      }

   return NULL;
}

int check_for_body( struct ccb *item )
{
   char *ptr;

   if (( ptr = get_var( "CONTENT_LENGTH", item->env )) == NULL )
      return 0;

   if ( strtol( ptr, NULL, 10 ))
      return 1;

   return 0;
}

char *form_decode( char *str )
{
   static struct string *s = NULL;
   char *ptr;

   if ( s == NULL )
      s = make_string();
   else
      STRING_TRUNCATE( s )

   for( ptr = str; *ptr; ++ptr )
      switch( *ptr )
      {
         case '+':
            STRING_APPEND( s, ' ' )
            break;

         case '%':
            if ( ! *( ptr + 1 ) || ! *( ptr + 2 ))
               STRING_APPEND( s, *ptr )
            else
            {
               char data[ 3 ];
               char code;

               data[ 0 ] = *( ptr + 1 );
               data[ 1 ] = *( ptr + 2 );
               data[ 2 ] = '\0';

               if (( code = ( char )strtol( data, NULL, 16 )))
               {
                  STRING_APPEND( s, code )
                  ptr += 2;
               }
               else
                  STRING_APPEND( s, *ptr )
            }
            break;

         default:
            STRING_APPEND( s, *ptr )
      }

   return str_dup( s->str, s->used );
}

void process_params( struct ccb *item )
{
   char *query, *ptr, *param, *new;

   query = get_var( "QUERY_STRING", item->env );

   if ( query == NULL || ! *query )
   {
      stack_push( item->params, NULL );
      return;
   }
   else
      new = query = str_dup( query, -1 );

   for( ptr = query; *ptr; ++ptr )
   {
      if ( *ptr == '&' )
      {
         *ptr = '\0';

         for( param = query; *param && *param != '='; ++param )
            ;

         if ( *param )
            *param++ = '\0';

         STACK_PUSH( item->params, form_decode( query ))
         STACK_PUSH( item->params, form_decode( param ))

         query = ptr + 1;
      }
   }

  if ( *query )
  {
      for( param = query; *param && *param != '='; ++param )
         ;

      if ( *param )
         *param++ = '\0';

      STACK_PUSH( item->params, form_decode( query ))
      STACK_PUSH( item->params, form_decode( param ))
   }

   STACK_PUSH( item->params, NULL )
   free( new );
   return;
}

void read_event( struct ccb *item )
{
   char *ptr, c;
   int n;

   if ( item->count >= 0 )
   {
      for( ptr = &item->buffer[ item->count ]; item->count < sizeof( item->buffer ); ++ptr, ++item->count )
      {
         if (( n = read( item->sock, &c, 1 )) <= 0 )
         {
            if ( n < 0 && errno == EWOULDBLOCK )
               return;

            remove_conn( item );
            return;
         }

         if ( c == ':' )
         {
            item->count = 0;
            *ptr = '\0';
            break;
         }

         *ptr = c;
      }

      if ( c != ':' )
      {
         remove_conn( item );
         return;
      }
   }

   if ( ! item->count )
   {
      if (( item->count = strtol( item->buffer, NULL, 10 )) <= 0 || item->count > 4096 )
      {
         remove_conn( item );
         return;
      }

      item->header = memory( item->count + 1 );
      item->count = -( item->count + 1 );
   }

   if ( item->count < 0 )
   {
      if (( n = read( item->sock, &item->header[ item->written ], -item->count )) <= 0 )
      {
         if ( n < 0 && errno == EWOULDBLOCK )
            return;

         remove_conn( item );
         return;
      }

      item->count += n;
      item->written += n;

      if ( item->count )
         return;
   }

   if ( item->header[ item->written - 1 ] != ',' )
   {
      remove_conn( item );
      return;
   }

   init_env( item->header, item->written - 1, item->env );
   item->count = item->written = 0;

   if ( check_for_body( item ))
   {
      remove_conn( item );
      return;
   }

   process_params( item );

   ev_set( item->sock, EVFILT_READ, EV_DELETE, item );
   ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
}

void add_headers( struct ccb *item )
{
   char links[ 512 ], js[ 8192 ], buffer[ IOSIZE ], *form, *javascript, *ptr;

   snprintf( js, sizeof( js ),
      "<script type=\"text/javascript\">\n"
      "var ephemera = function() {\n"
      "var content, callink, xhr, request = '';\n"

      "function init( e )\n"
      "{\n"
      "   if ( typeof( document.forms.search ) != 'undefined' )\n"
      "      document.forms.search.key.addEventListener( 'click', function( e ) { document.forms.search.key.select(); }, false );\n"
      "   xhr = new XMLHttpRequest();\n"
      "   xhr.onreadystatechange = fade_in;\n"
      "   callink = document.getElementById( 'callink' );\n"
      "   content = document.getElementById( 'content' );\n"
      "   content.style.transition = 'opacity 0.1s ease-in';\n"
      "   content.addEventListener( 'transitionend', load_new, false );\n"
      "}\n"

      "function fade_out_search()\n"
      "{\n"
      "   request = 'request=search&key=' + encodeURI( document.forms.search.key.value );\n"
      "   content.style.opacity = 0;\n"
      "}\n"

      "function fade_out( r )\n"
      "{\n"
      "   if ( typeof( r ) != 'undefined' )\n"
      "   {\n"
      "      if ( r.startsWith( 'request=calendar', 0 ))\n"
      "      {\n"
      "         if ( ! request.startsWith( 'request=calendar', 0 ) && ! request.startsWith( 'request=day', 0 ) && ! request.startsWith( 'request=month', 0 ))\n"
      "         {\n"
      "            callink.href = 'javascript:ephemera.load(\"' + request + '\")';\n"
      "            callink.innerHTML = 'Saved';\n"
      "         }\n"
      "      }\n"
      "      else if ( ! r.startsWith( 'request=day', 0 ) && ! r.startsWith( 'request=month', 0 ) && callink.innerHTML != 'Year' )\n"
      "      {\n"
      "         callink.href = 'javascript:ephemera.load( \"request=calendar\" )';\n"
      "         callink.innerHTML = 'Year';\n"
      "      }\n"
      "   }\n"
      "   request = r;\n"
      "   content.style.opacity = 0;\n"
      "}\n"

      "function load_new( e )\n"
      "{\n"
      "   if ( content.style.opacity == 1 )\n"
      "      return;\n"
      "   try {\n"
      "      var resource = 'https://' + window.location.hostname + '%s?xhr' + ( request == '' ? '' : '&' ) + request;\n"
      "      xhr.open( 'GET', resource, true )\n"
      "      xhr.send();\n"
      "   } catch( e ) { window.alert( e.message ); }\n"
      "}\n"

      "function fade_in( e )\n"
      "{\n"
      "   if ( xhr.readyState != 4 )\n"
      "      return;\n"
      "   if ( xhr.status != 200 )\n"
      "   {\n"
      "      window.alert( xhr.statusText );\n"
      "      return;\n"
      "   }\n"
      "   content.innerHTML = xhr.responseText;\n"
      "   content.style.opacity = 1;\n"
      "}\n"

      "document.addEventListener( 'DOMContentLoaded', init, false );\n"
      "return { load : fade_out, search : fade_out_search };\n"
      "}();\n"
      "</script>\n",
      path );

   if ( permalink )
   {
      snprintf( links, sizeof( links ), "<div id=\"links\">\n<p><a href=\"%s\">New Posts</a></p>\n</div>\n", path );
      javascript = form = "";
   }
   else
   {
      snprintf( links, sizeof( links ), "%s",
         "<div id=\"links\">\n"
         "<p><a href=\"javascript:ephemera.load('')\">New</a> | "
         "<a href=\"javascript:ephemera.load('request=oldest')\">Old</a> | "
         "<a id=\"callink\" href=\"javascript:ephemera.load('request=calendar')\">Year</a>"
         "</p></div>\n" );

      form = "<div id=\"form\">\n"
               "<form id=\"search\" name=\"search\" action=\"javascript:ephemera.search()\" method=\"GET\" enctype=\"x-www-form-urlencoded\">\n"
               "<input id=\"key\" type=\"text\" name=\"key\" value=\"SEARCH\" />\n"
               "<input type=\"hidden\" name=\"request\" value=\"search\" />\n"
               "</form>\n"
            "</div>\n";

      javascript = js;
   }

   snprintf( buffer, sizeof( buffer ), "Content-Type: text/html; charset=utf-8\r\n"
      "Cache-Control: no-store, no-cache, must-revalidate, max-age=0\r\n\r\n"

      "<!DOCTYPE html>\n"
      "<html lang=\"en\">\n"
      "<head>\n"
      "<meta charset=\"UTF-8\">\n"
      "<meta name=\"generator\" content=\"ephemera " VERSION "\">\n"
      "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">\n"

      "%s"
      "%s"
      "%s"
      "%s"
      "%s"
      "%s"

      "<title>%s</title>\n"

      "%s"

      "</head>\n"
      "<body>\n"
      "<h1>%s</h1>\n"
      "%s"
      "<div id=\"header\">\n"
      "%s"
      "%s"
      "</div>\n"
      "<div id=\"main\">\n"
      "<div id=\"content\">\n",

      css, desc, keys, lang, robs, fav,
      title, javascript, title, blb,
      links,
      form );

   for( ptr = buffer; *ptr; ++ptr )
      STRING_APPEND( item->response, *ptr )
}

void add_error( struct ccb *item )
{
   char *msg = "<div id=\"error\"><p>[ internal error: error message logged ]</p></div>\n",
        *ptr;

   for( ptr = msg; *ptr; ++ptr )
      STRING_APPEND( item->response, *ptr )
}

struct string *add_each( struct ccb *item, sqlite3_stmt *sql, int idx, int *offset )
{
   static struct stack *row = NULL;
   static struct string *local = NULL;
   char buffer[ 4096 ], perma[ 1024 ], *ptr;
   int i;

   *perma = '\0';

   if ( row == NULL )
      row = make_stack();

   if ( local == NULL )
      local = make_string();
   else
   {
      local->free += local->used;
      local->used = 0;
      local->top = local->str;
   }

   for( ; ; )
   {
      if ( sqlite_row( sql, &row ) < 0 )
      {
         add_error( item );
         return NULL;
      }

      ++idx;

      if ( ! permalink )
         snprintf( perma, sizeof( perma ),
                   "<div class=\"permalink\">"
                   "<p><a href=\"%s?request=one&amp;id=%s\">permalink</a></p>"
                   "</div>\n",
                   path, ( char *)row->values[ 3 ] );

      snprintf( buffer, sizeof( buffer ),
                "<div class=\"article\">\n"
                "<!-- %s -->\n"
                "%s"
                "<div class=\"date\"><p>%s</p></div>\n"
                "<div class=\"internal\">\n"
                "<h3>%s</h3>\n",

                row->values[ 3 ],
                perma,
                row->values[ 0 ],
                row->values[ 1 ] );

      for( ptr = buffer; *ptr; ++ptr )
         STRING_APPEND( local, *ptr )

      for( ptr = row->values[ 2 ]; *ptr; ++ptr )
         STRING_APPEND( local, *ptr )

      for( i = 0; i < 2; ++i )
         for( ptr = "</div>\n"; *ptr; ++ptr )
            STRING_APPEND( local, *ptr );

      free( row->values[ 0 ] );
      free( row->values[ 1 ] );
      free( row->values[ 2 ] );
      free( row->values[ 3 ] );

      row->free += row->used;
      row->used = 0;
      row->top = row->values;

      if (( i = sqlite_step( sql )) <= 0 )
         break;
   }

   if ( i < 0 )
   {
      add_error( item );
      return NULL;
   }

   *offset = idx;
   return local;
}

void add_nav( struct ccb *item, sqlite3_stmt *sql, char *h2, char *request, int old_offset, int *offset, struct string *result )
{
   char buffer[ 4096 ], before[ 1024 ], after[ 1024 ], *ptr;
   int tmp;

   if ( permalink )
   {
      if ( result != NULL )
         for( ptr = result->str; *ptr; ++ptr )
            STRING_APPEND( item->response, *ptr )

      return;
   }

   *buffer = *before = *after = '\0';

   if (( tmp = old_offset - p_size ) < 0 )
      tmp = 0;

   snprintf( before, sizeof( before ), "<p><a href=\"javascript:ephemera.load('request=%s&amp;index=%d')\">&lt;&lt;&lt;</a></p>", request, tmp );
   snprintf( after, sizeof( after ), "<p><a href=\"javascript:ephemera.load('request=%s&amp;index=%d')\">&gt;&gt;&gt;</a></p>", request, *offset );

   snprintf( buffer, sizeof( buffer ),
          "<div class=\"pager\">\n"
          "<div class=\"pager_left\">%s</div>"
          "<div class=\"pager_center\">%s</div>"
          "<div class=\"pager_right\">%s</div>\n"
          "</div>\n",
          ( tmp == old_offset ? "<p>&lt;&lt;&lt;</p>" : before ),
          h2,
          (( result == NULL || abs(old_offset - *offset) < p_size ) ? "<p>&gt;&gt;&gt;</p>" : after ));

   for( ptr = buffer; *ptr; ++ptr )
      STRING_APPEND( item->response, *ptr )

   if ( result != NULL )
   {
      for( ptr = result->str; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )

      for( ptr = buffer; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )
   }
}

void add_results( struct ccb *item, char *h2, sqlite3_stmt *sql, char *request, int offset )
{
   char buffer[ 512 ], *ptr;
   int result, new_offset = 0, header = 0;

   if ( sql == recent_sql && recent == NULL && ! offset )
      header = item->response->used;

   if (( result = sqlite_step( sql )) < 0 )
      add_error( item );
   else if ( result )
      add_nav( item, sql, h2, request, offset, &new_offset, add_each( item, sql, offset, &new_offset ));
   else
   {
      snprintf( buffer, sizeof( buffer ),
               "<div id=\"nomore\"><p>No %sposts.</p></div>",
               ( offset ? "more " : "" ));

      if ( offset )
         add_nav( item, sql, h2, request, offset, &new_offset, NULL );

      for( ptr = buffer; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )
   }

   sqlite_reset( sql );

   if ( sql == recent_sql && recent == NULL && ! offset )
      recent = str_dup( &item->response->str[ header ], item->response->used - header );
}

void add_recent( struct ccb *item )
{
   int idx;
   char *ptr;

   if (( ptr = get_var( "index", item->params )) == NULL )
   {
      idx = 0;
      ptr = "0";
   }
   else if (( idx = strtol( ptr, NULL, 10 )) < 0 )
   {
      idx = 0;
      ptr = "0";
   }

   if ( ! idx && recent != NULL )
   {
      for( ptr = recent; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )
   }
   else
   {
      if ( sqlite_bind( recent_sql, 1, page_size, -1 ) < 0 ||
           sqlite_bind( recent_sql, 2, ptr, -1 ) < 0 )
      {
         add_error( item );
         sqlite_reset( recent_sql );
         return;
      }

      add_results( item, "New Posts", recent_sql, "recent", idx );
   }
}

void add_oldest( struct ccb *item )
{
   int idx;
   char *ptr;

   if (( ptr = get_var( "index", item->params )) == NULL )
   {
      idx = 0;
      ptr = "0";
   }
   else if (( idx = strtol( ptr, NULL, 10 )) < 0 )
   {
      idx = 0;
      ptr = "0";
   }

   if ( sqlite_bind( oldest_sql, 1, page_size, -1 ) < 0 ||
        sqlite_bind( oldest_sql, 2, ptr, -1 ) < 0 )
   {
      add_error( item );
      sqlite_reset( oldest_sql );
      return;
   }

   add_results( item, "Old Posts", oldest_sql, "oldest", idx );
}

void add_one( struct ccb *item )
{
   char *id;

   if (( id = get_var( "id", item->params )) == NULL )
   {
      add_recent( item );
      return;
   }

   if ( sqlite_bind( one_sql, 1, id, -1 ) < 0 )
   {
      add_error( item );
      sqlite_reset( one_sql );
      return;
   }

   add_results( item, "Permalink", one_sql, NULL, 0 );
}

void add_month( struct ccb *item )
{
   char *t, *eom, *ptr, *month, mbuffer[ 128 ];
   int idx;

   if (( t = get_var( "time", item->params )) == NULL )
   {
      add_recent( item );
      return;
   }

   if (( ptr = get_var( "index", item->params )) == NULL )
   {
      idx = 0;
      ptr = "0";
   }
   else if (( idx = strtol( ptr, NULL, 10 )) < 0 )
   {
      idx = 0;
      ptr = "0";
   }

   if (( eom = get_eom( t, &month )) == NULL )
   {
      add_error( item );
      return;
   }

   if ( sqlite_bind( era_sql, 1, t, -1 ) < 0 ||
        sqlite_bind( era_sql, 2, eom, -1 ) < 0 ||
        sqlite_bind( era_sql, 3, page_size, -1 ) < 0 ||
        sqlite_bind( era_sql, 4, ptr, -1 ) < 0 )
   {
      add_error( item );
      sqlite_reset( era_sql );
      return;
   }

   snprintf( mbuffer, sizeof( mbuffer ), "month&amp;time=%s", t );
   add_results( item, month, era_sql, mbuffer, idx );
}

void add_day( struct ccb *item )
{
   char *t, *eod, *ptr, *day, dbuffer[ 128 ];
   int idx;

   if (( t = get_var( "time", item->params )) == NULL )
   {
      add_recent( item );
      return;
   }

   if (( ptr = get_var( "index", item->params )) == NULL )
   {
      idx = 0;
      ptr = "0";
   }
   else if (( idx = strtol( ptr, NULL, 10 )) < 0 )
   {
      idx = 0;
      ptr = "0";
   }

   if (( eod = get_eod( t, &day )) == NULL )
   {
      add_error( item );
      return;
   }

   if ( sqlite_bind( era_sql, 1, t, -1 ) < 0 ||
        sqlite_bind( era_sql, 2, eod, -1 ) < 0 ||
        sqlite_bind( era_sql, 3, page_size, -1 ) < 0 ||
        sqlite_bind( era_sql, 4, ptr, -1 ) < 0 )
   {
      add_error( item );
      sqlite_reset( era_sql );
      return;
   }

   snprintf( dbuffer, sizeof( dbuffer ), "day&amp;time=%s", t );
   add_results( item, day, era_sql, dbuffer, idx );
}

void free_times( char **times )
{
   char **ptr;

   for( ptr = times; *ptr != NULL; ++ptr )
      free( *ptr );
}

void add_calendar( struct ccb *item )
{
   char *ptr;
   int this_year, year;

   this_year = get_year();

   if (( ptr = get_var( "year", item->params )) == NULL )
      year = this_year;
   else
      year = strtol( ptr, NULL, 10 );

   if ( year > this_year )
   {
      char *msg = "<div id=\"error\"><p>Ephemera cannot display calendars for years beyond the current year.</p></div>\n";

      for( ptr = msg; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )

      return;
   }

   if ( year < 2010 )
   {
      char *msg = "<div id=\"error\"><p>Ephemera cannot display calendars for years before 2010.</p></div>\n";

      for( ptr = msg; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )

      return;
   }

   /*
    * The previous 2 tests ensure that the year is within the range
    * generated by make_calendar_cache(), but if the year has changed since
    * the server started, the cache must be regenerated to include the
    * calendar for the new year.
    */

   if ( calendars->used <= ( year - 2010 ))
      make_calendar_cache();

   for( ptr = calendars->values[ year - 2010 ]; *ptr; ++ptr )
      STRING_APPEND( item->response, *ptr );
}

char *compose_calendar( int this_year, int year )
{
   int mon, i, count, num_active;
   char **times, *active[ 32 ], **tptr, *ptr, *cal;
   char header[ 4096 ], before[ 1024 ], after[ 1024 ];
   struct stack *row;
   struct string *result;

   row = make_stack();
   result = make_string();

   snprintf( before, sizeof( before ), "<div class=\"pager_left\"><p><a href=\"javascript:ephemera.load('request=calendar&amp;year=%d')\">&lt;&lt;&lt;</a></p></div>", year - 1 );
   snprintf( after, sizeof( after ), "<div class=\"pager_right\"><p><a href=\"javascript:ephemera.load('request=calendar&amp;year=%d')\">&gt;&gt;&gt;</a></p></div>\n", year + 1 );

   snprintf( header, sizeof( header ),
             "<div class=\"pager\">\n"
             "%s"
             "<div class=\"pager_center\"><p>%d</p></div>"
             "%s"
             "</div>\n",
             ( year < 2011 ? "<div class=\"pager_left\"><p>&lt;&lt;&lt;</p></div>" : before ),
             year,
             ( year == this_year ? "<div class=\"pager_right\"><p>&gt;&gt;&gt;</p></div>\n" : after ));

   for( ptr = header; *ptr; ++ptr )
      STRING_APPEND( result, *ptr )

   for( ptr = "<div id=\"caldiv\">\n"; *ptr; ++ptr )
      STRING_APPEND( result, *ptr )

   for( mon = 1; mon < 13; ++mon )
   {
      num_active = 0;

      if (( times = make_active_times( year, mon )) == NULL )
         return NULL;

      for( i = 0, tptr = times; i < 32 && *tptr != NULL; tptr += 2, ++i )
      {
         if ( sqlite_bind( count_sql, 1, *tptr, -1 ) < 0 ||
              sqlite_bind( count_sql, 2, *( tptr + 1), -1 ) < 0 )
            return NULL;

         if (( count = sqlite_step( count_sql )) < 0 )
            return NULL;
         else if ( count )
         {
            if ( sqlite_row( count_sql, &row ) < 0 )
               return NULL;

            if ( *row->values[ 0 ] == '0' )
               active[ i ] = NULL;
            else
            {
               ++num_active;
               active[ i ] = *tptr;
            }

            free( row->values[ 0 ] );

            row->free += row->used;
            row->used = 0;
            row->top = row->values;
         }
         else
            active[ i ] = NULL;

         sqlite_reset( count_sql );
      }

      if (( cal = make_calendar( year, mon, num_active, active, path )) == NULL )
         return NULL;

      free_times( times );

      for( ptr = cal; *ptr; ++ptr )
         STRING_APPEND( result, *ptr )
   }

   for( ptr = "</div>\n</div>\n"; *ptr; ++ptr )
      STRING_APPEND( result, *ptr )

   for( ptr = header; *ptr; ++ptr )
      STRING_APPEND( result, *ptr )

   ptr = str_dup( result->str, result->used );
   string_free( result );
   stack_free( row );

   return ptr;
}

void make_calendar_cache()
{
   char *ptr, *msg = "<div id=\"error\"><p>[ internal error: error message logged ]</p></div>\n";
   int this_year, y;

   if ( calendars == NULL )
      calendars = make_stack();
   else
      while( calendars->used )
         free( stack_pop( calendars ));

   this_year = get_year();

   for( y = 2010; y <= this_year; ++y )
   {
      if (( ptr = compose_calendar( this_year, y )) == NULL )
      {
         STACK_PUSH( calendars, str_dup( msg, -1 ))
         continue;
      }

      STACK_PUSH( calendars, ptr );
   }
}

char *form_encode( char *str )
{
   char *ptr, *ptr2;
   static struct string *s = NULL;
   int found;
   static char *reserved = "']$;:@&+!?=#{}/[^`~\"<>|%\\\t\r\n",
               *letters  = "0123456789ABCDEF";

   if ( s == NULL )
      s = make_string();
   else
      STRING_TRUNCATE( s )

   for( ptr = str; *ptr; ++ptr )
   {
      found = 0;

      if ( *ptr == ' ' )
      {
         STRING_APPEND( s, '+' )
         continue;
      }

      if ( *ptr < 32 || *ptr > 126 )
      {
         STRING_APPEND( s, '%' )
         STRING_APPEND( s, letters[ *ptr / 16 ] )
         STRING_APPEND( s, letters[ *ptr % 16 ] )
         continue;
      }

      for( ptr2 = reserved; *ptr2; ++ptr2 )
         if ( *ptr == *ptr2 )
         {
            STRING_APPEND( s, '%' )
            STRING_APPEND( s, letters[ *ptr / 16 ] )
            STRING_APPEND( s, letters[ *ptr % 16 ] )
            found = 1;
            break;
         }

      if ( ! found )
         STRING_APPEND( s, *ptr )
   }

   return s->str;
}

void add_search( struct ccb *item )
{
   char *key, *ptr, request[ 256 ];
   int idx;

   if (( key = get_var( "key", item->params )) == NULL )
   {
      syslog( LOG_ERR, "search request without key" );
      add_error( item );
      return;
   }

   if (( ptr = get_var( "index", item->params )) == NULL )
   {
      idx = 0;
      ptr = "0";
   }
   else if (( idx = strtol( ptr, NULL, 10 )) < 0 )
   {
      idx = 0;
      ptr = "0";
   }

   if ( sqlite_bind( search_sql, 1, key, -1 ) < 0       ||
        sqlite_bind( search_sql, 2, page_size, -1 ) < 0 ||
        sqlite_bind( search_sql, 3, ptr, -1 ) < 0 )
   {
      add_error( item );
      sqlite_reset( search_sql );
      return;
   }

   snprintf( request, sizeof( request ), "search&key=%s", form_encode( key ));
   add_results( item, "Search Results", search_sql, request, idx );
}

void add_response( struct ccb *item )
{
   char *req, *ptr1, *ptr2, **ptr;

   static char *values[] = {
      "recent", "oldest", "one", "search", "month", "day", "calendar", NULL
   };

   static void ( *funcs[] )( struct ccb * ) = {
      &add_recent, &add_oldest, &add_one, &add_search, &add_month, &add_day, &add_calendar, NULL
   };

   void ( **fptr )( struct ccb * );

   if (( req = get_var( "request", item->params )) == NULL )
      add_recent( item );
   else
   {
      for( ptr = values, fptr = funcs; *fptr != NULL; ++ptr, ++fptr )
      {
         ptr1 = *ptr;
         ptr2 = req;

         while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
         {
            ++ptr1;
            ++ptr2;
         }

         if ( ! *ptr1 && ! *ptr2 )
            break;
      }

      if ( *fptr == NULL )
         fptr = &funcs[ 0 ];

      ( *fptr )( item );
   }

   return;
}

void set_permalink( struct ccb *item )
{
   char *ptr1, *ptr2;

   if (( ptr1 = get_var( "request", item->params )) == NULL )
      return;

   ptr2 = "one";

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      permalink = 1;
}

int process_request( struct ccb *item )
{
   char *ptr;
   int xhr;

   xhr = ( get_var( "xhr", item->params ) != NULL );

   if ( xhr )
      for ( ptr = "Content-Type: text/plain; charset=utf-8\r\n"
                  "Cache-Control: no-store, must-revalidate, max-age=0\r\n\r\n"; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )
   else
   {
      set_permalink( item );
      add_headers( item );
   }

   add_response( item );

   if ( ! xhr )
   {
      for( ptr = "</div>\n</div>\n</body>\n</html>\n"; *ptr; ++ptr )
         STRING_APPEND( item->response, *ptr )

      permalink = 0;
   }

   return item->response->used;
}

void sigterm_handler( int signo )
{
   terminate = 1;
}

void reload_handler( int signo )
{
   reload = 1;
}

void set_signal_intr( int signo, void ( *handler )( int ))
{
   struct sigaction sigact;

   sigact.sa_handler = handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( signo, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "initialize: sigaction: %m" );
}

void set_signals()
{
   int *iptr;
   int sigs[] = {
      SIGCHLD, SIGPIPE, SIGQUIT, SIGUSR1, SIGUSR2, SIGALRM, SIGINT, SIGTSTP, -1
   };

   if ( ! testing )
      for( iptr = sigs; *iptr > 0; ++iptr )
         signal( *iptr, SIG_IGN );

   signal( SIGHUP,  reload_handler );
   signal( SIGTERM, sigterm_handler );
}

void finalize_sql()
{
   if ( recent_sql != NULL )
      sqlite3_finalize( recent_sql );

   if ( oldest_sql != NULL )
      sqlite3_finalize( oldest_sql );

   if ( one_sql != NULL )
      sqlite3_finalize( one_sql );

   if ( search_sql != NULL )
      sqlite3_finalize( search_sql );

    if ( era_sql != NULL )
      sqlite3_finalize( era_sql );

   if ( count_sql != NULL )
      sqlite3_finalize( count_sql );
}

void compile_sql()
{
   recent_sql = sqlite_compile( db,
      "SELECT Date,Title,Body,Id FROM Articles ORDER BY Time DESC LIMIT ?001 OFFSET ?002", -1 );

   if ( recent_sql == NULL )
      exit( 1 );

   oldest_sql = sqlite_compile( db,
      "SELECT Date,Title,Body,Id FROM Articles ORDER BY Time LIMIT ?001 OFFSET ?002", -1 );

   if ( oldest_sql == NULL )
      exit( 1 );

   one_sql = sqlite_compile( db, "SELECT Date,Title,Body,Id FROM Articles WHERE Id = ?", -1 );

   if ( one_sql == NULL )
      exit( 1 );

   search_sql = sqlite_compile( db,
      "SELECT Date,Title,Body,Id FROM Search WHERE Search MATCH ?001 ORDER BY Time DESC LIMIT ?002 OFFSET ?003", -1 );

   if ( search_sql == NULL )
      exit( 1 );

   era_sql = sqlite_compile( db,
       "SELECT Date,Title,Body,Id FROM Articles WHERE Time >= ?001 AND Time <= ?002 ORDER BY Time DESC LIMIT ?003 OFFSET ?004", -1 );

   if ( era_sql == NULL )
      exit( 1 );

   count_sql = sqlite_compile( db, "SELECT count(*) FROM Articles WHERE Time >= ?001 AND Time <= ?002", -1 );

   if ( count_sql == NULL )
      exit( 1 );
}

void create_virtual_table()
{
   if ( sqlite_exec( db,
         "CREATE VIRTUAL TABLE Search USING fts4 ( content=Articles, tokenize=porter, "
         "Id STRING UNIQUE, Date TEXT, Title TEXT, Body TEXT, Time INTEGER ); "

         "CREATE TRIGGER s_bu BEFORE UPDATE ON Articles BEGIN "
         "DELETE FROM Search WHERE docid=old.rowid; "
         "END; "

         "CREATE TRIGGER s_bd BEFORE DELETE ON Articles BEGIN "
         "DELETE FROM Search WHERE docid=old.rowid; "
         "END; "

         "CREATE TRIGGER s_au AFTER UPDATE ON Articles BEGIN "
         "INSERT INTO Search( docid, Date, Title, Body, Time ) "
         "VALUES( new.rowid, new.Date, new.Title, new.Body, new.Time ); "
         "END; "

         "CREATE TRIGGER s_ai AFTER INSERT ON Articles BEGIN "
         "INSERT INTO Search( docid, Date, Title, Body, Time ) "
         "VALUES( new.rowid, new.Date, new.Title, new.Body, new.Time ); "
         "END;" ))
   exit( 1 );
}

void open_db()
{
   char buffer[ PATH_MAX ];
   sqlite3 *ddb;
   sqlite3_backup *backup;

   reload = 0;

   if ( db != NULL )
   {
      finalize_sql();
      sqlite3_close( db );

      if ( recent != NULL )
      {
         free( recent );
         recent = NULL;
      }
   }

   snprintf( buffer, sizeof( buffer ), "%s/ephemera.sqlite", directory );

   if ( ! in_memory )
   {
      if (( db = sqlite_open( buffer )) == NULL )
         exit( 1 );
   }
   else
   {
      if (( ddb = sqlite_open( buffer )) == NULL )
         exit( 1 );
   
      if (( db = sqlite_open( ":memory:" )) == NULL )
         exit( 1 );
      
      if (( backup = sqlite3_backup_init( db, "main", ddb, "main" )) == NULL )
      {
         syslog( LOG_ERR, "sqlite3_backup_init(): %s", ( char *)sqlite3_errmsg( db ));
         exit( 1 );
      }
   
      sqlite3_backup_step( backup, -1 );
      sqlite3_backup_finish( backup );
      sqlite3_close( ddb );
   }

   compile_sql();
   make_calendar_cache();
}

void init_db()
{
   if ( sqlite_exec( db,
            "CREATE TABLE Articles "
            "( Id STRING UNIQUE, Date TEXT, Title TEXT, Body TEXT, Time INTEGER ); "
            "CREATE INDEX t_idx ON Articles ( Time ); " ))

      exit( 1 );

   create_virtual_table();
}

void create_db()
{
   char buffer[ PATH_MAX ];
   struct stat st;

   snprintf( buffer, sizeof( buffer ), "%s/ephemera.sqlite", directory );

   if ( stat( buffer, &st ) ==  0 )
      return;

   if ( errno != ENOENT )
   {
      fprintf( stderr, "ephemera: stat( %s ): %s\n", buffer, strerror( errno ));
      exit( 1 );
   }

   if (( db = sqlite_open( buffer )) == NULL )
      exit( 1 );

   init_db();
   sqlite3_close( db );
}

void write_event( struct ccb *item )
{
   off_t written;

   if ( ! item->count )
   {
      if (( item->count = process_request( item )) <= 0 )
      {
         remove_conn( item );
         return;
      }
   }

   written = write( item->sock, &item->response->str[ item->written ], MIN( item->count, IOSIZE ));

   if ( written < 0 )
   {
      if ( errno != EWOULDBLOCK )
         remove_conn( item );

      return;
   }

   item->count -= written;
   item->written += written;

   if ( ! item->count )
      remove_conn( item );
}

/*
 * If we have dropped a connection, we traverse the output kqueue
 * and mark any events for that connection as invalid, by setting
 * the socket descriptor to 0.
 */

void remove_events( struct ccb *item, int n )
{
   struct kevent *eptr;

   for( eptr = &outqueue[ ++n ]; n < out; ++n, ++eptr )
      if ( eptr->udata == item )
         eptr->ident = 0;
}

void process_clients()
{
   int kq, n;
   struct kevent *eptr;

   if (( inqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( inqueue, sizeof( struct kevent ) * qlen );
   
   if (( outqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( outqueue, sizeof( struct kevent ) * qlen );

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   ev_set( fd, EVFILT_READ, EV_ADD | EV_ENABLE, NULL );

   for( ; ; )
   {
      if ( terminate )
      {
         if ( fd > 0 )
         {
            close( fd );
            fd = -1;
         }

         if ( ! active )
            exit( 0 );
      }

      if ( reload )
         open_db();

      set_signal_intr( SIGHUP, reload_handler );
      set_signal_intr( SIGTERM, sigterm_handler );

      out = kevent( kq, inqueue, in, outqueue, qlen, NULL );
      in = 0;

      if ( out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %m" );
         exit( 1 );
      }

      signal( SIGHUP, reload_handler );
      signal( SIGTERM, sigterm_handler );

      for( n = 0, eptr = &outqueue[ n ]; n < out; ++n, ++eptr )
      {
         struct ccb *item = ( struct ccb *)eptr->udata;

         if ( ! eptr->ident || eptr->flags & EV_ERROR )
            continue;

         closed = 0;

         if ( eptr->ident == fd )
         {
            accept_connection();
            continue;
         }

         switch( eptr->filter )
         {
            case EVFILT_READ:
               read_event( item );
               break;

            case EVFILT_WRITE:
               write_event( item );
               break;

            default:
               continue;
         }

         if ( closed )
            remove_events( item, n );
      }
   }
}

void set_variable( char *variable, char *value )
{

   char *ptr1, *ptr2, **vptr, ***aptr;

   for( vptr = var_names, aptr = var_ptrs; *vptr != NULL; ++vptr, ++aptr )
   {
      ptr1 = *vptr;
      ptr2 = variable;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 && ! *ptr2 )
      {
         **aptr = str_dup( value, -1 );
         break;
      }
   }

   if ( *vptr == NULL )
   {
      fprintf( stderr, "ephemera: unrecognized configuration variable: %s\n", variable );
      exit( 1 );
   }
}

void make_static()
{
   snprintf( css, sizeof( css ),
      "<link rel=\"stylesheet\" media=\"screen\" href=\"%sephemera.css\" type=\"text/css\">\n",
      web_directory );

   if ( meta_description == NULL )
      *desc = '\0';
   else
      snprintf( desc, sizeof( desc ), "<meta name=\"description\" content=\"%s\">\n", meta_description );

   if ( meta_keywords == NULL )
      *keys = '\0';
   else
      snprintf( keys, sizeof( keys ), "<meta name=\"keywords\" content=\"%s\">\n", meta_keywords );

   if ( meta_language == NULL )
      *lang = '\0';
   else
      snprintf( lang, sizeof( lang ), "<meta name=\"language\" content=\"%s\">\n", meta_language );

   if ( meta_robots == NULL )
      *robs = '\0';
   else
      snprintf( robs, sizeof( robs ), "<meta name=\"robots\" content=\"%s\">\n", meta_robots );

   if ( favicon == NULL )
      *fav = '\0';
   else
      snprintf( fav, sizeof( fav ), "<link rel=\"shortcut icon\" href=\"%s%s\">\n", web_directory, favicon );

   if ( blurb == NULL )
      *blb = '\0';
   else
      snprintf( blb, sizeof( blb ), "<div id=\"blurb\"><p>%s</p></div>\n", blurb );
}

void check_vars()
{
   char **aptr, ***vptr, *lptr;
   struct stat st;
   int len;

   if ( page_size == NULL )
      p_size = 5;
   else
      p_size = strtol( page_size, NULL, 10 );

   if ( web_directory == NULL )
   {
      fputs( "ephemera: \"web_directory\" not specified in ephemera.config\n", stderr );
      exit( 1 );
   }

   if ( directory == NULL )
   {
      fputs( "ephemera: \"directory\" not specified in ephemera.config\n", stderr );
      exit( 1 );
   }

   if ( stat( directory, &st ) < 0 )
   {
      if ( errno == ENOENT || errno == ENOTDIR )
         fprintf( stderr, "ephemera: database directory %s does not exist\n", directory );
      else
         fprintf( stderr, "ephemera: stat( %s ): %s\n", directory, strerror( errno ));

      exit( 1 );
   }

   if ( path == NULL )
   {
      fputs( "ephemera: \"path\" not specified in ephemera.config\n", stderr );
      exit( 1 );
   }

   if ( title == NULL )
   {
      fputs( "ephemera: \"title\" not specified in ephemera.config\n", stderr );
      exit( 1 );
   }

   for( aptr = var_names, vptr = var_ptrs; *aptr != NULL; ++aptr, ++vptr )
      if ( **vptr != NULL )
         for( lptr = **vptr, len = 0; *lptr; ++lptr )
            if ( ++len > VARLEN )
            {
               fprintf( stderr, "ephemera: \"%s\" is too long (max len = %d)\n", *aptr, VARLEN );
               exit( 1 );
            }

   make_static();
}

void set_config()
{
   FILE *config;
   char buffer[ PATH_MAX ];
   int line;

   snprintf( buffer, sizeof( buffer ), "%s/ephemera.config", config_dir );

   if (( config = fopen( buffer, "r" )) == NULL )
   {
      fprintf( stderr, "fopen( %s ): %s\n", buffer, strerror( errno ));
      exit( 1 );
   }

   line = 0;

   while( fgets( buffer, sizeof( buffer ), config ) != NULL )
   {
      char *ptr, *var, *val;

      ++line;

      if ( *buffer == ' ' || *buffer == '\t' || *buffer == ';' || *buffer == '\r' || *buffer == '\n' )
         continue;

      var = buffer;

      for( ptr = buffer; *ptr && *ptr != ':'; ++ptr )
         ;

      if ( *ptr != ':' )
      {
         fprintf( stderr, "ephemera: ':' missing from line %d in ephemera.config\n", line );
         exit( 1 );
      }

      *ptr = '\0';
      val = ++ptr;

      for( ; *ptr && *ptr != '\n' && *ptr != '\r'; ++ptr )
         ;

      *ptr = '\0';

      if ( *val )
         set_variable( var, val );
   }

   fclose( config );
   check_vars();
}

int main( int argc, char **argv )
{
   set_options( argc, argv );
   set_signals();
   set_config();

   create_db();

   openlog( "ephemera", LOG_PID, LOG_DAEMON );
   logging = 1;

   if ( !testing )
      become_daemon();

   if ( unix_socket == NULL )
      start_listening();
   else
      start_listening_unix();

   open_db();

   change_identity();
   process_clients();

   return 0;
}
