/*
 * Ephemera Copyright (c) 2014, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <time.h>
#include <calendar.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <syslog.h>

#include "data.h"

char *months[] = { "January", "February", "March", "April", "May", "June",
                   "July", "August", "September", "October", "November", "December" };

char *active_times[ 64 ], eod_buffer[ 64 ], day_buffer[ 128 ], eom_buffer[ 64 ], month_buffer[ 128 ];

int get_year()
{
   struct tm *tm;
   time_t t;

   t = time( NULL );

   if (( tm = localtime( &t )) == NULL )
   {
      syslog( LOG_ERR, "get_year(): localtime( &t ) failed\n" );
      return -1;
   }

   return tm->tm_year + 1900;
}

char **make_active_times( int year, int month )
{
   int len, i, day, idx, first, last;
   char buffer[ 128 ];
   struct date date;
   struct tm tm;
   time_t eod;

   date.y = year;
   date.m = month;
   date.d = 1;

   if (( first = ndaysg( &date )) < 0 )
   {
      syslog( LOG_ERR, "make_active_times(): ndaysg() returned an error" );
      return NULL;
   }

   if ( date.m < 12 )
      ++date.m;
   else
   {
      ++date.y;
      date.m = 1;
   }

   if (( last = ndaysg( &date )) < 0 )
   {
      syslog( LOG_ERR, "make_active_times(): ndaysg() returned an error" );
      return NULL;
   }

   for( i = 0; i < 64; ++i )
      active_times[ i ] = NULL;

   idx = 0;
   day = 0;

   tm.tm_year = year - 1900;
   tm.tm_mon = month - 1;

   for( i = first; i < last; ++i )
   {
      tm.tm_mday = ++day;
      tm.tm_hour = 0;
      tm.tm_min = 0;
      tm.tm_sec = 0;
      tm.tm_isdst = -1;

      if (( eod = mktime( &tm )) < 0 )
      {
         syslog( LOG_ERR, "make_active_times(): mktime() returned an error" );
         return NULL;
      }

      len = snprintf( buffer, sizeof( buffer ), "%lu", ( long unsigned int)eod );
      active_times[ idx++ ] = str_dup( buffer, len );

      tm.tm_hour = 23;
      tm.tm_min = 59;
      tm.tm_sec = 59;
      tm.tm_isdst = -1;

      /*
       * We can't just add 86399 to get the time of the end of the day because if a change
       * to or from daylight savings occurs on this day, it will not be 86400 seconds long.
       */

      if (( eod = mktime( &tm )) < 0 )
      {
         syslog( LOG_ERR, "make_active_times(): mktime() returned an error" );
         return NULL;
      }

      len = snprintf( buffer, sizeof( buffer ), "%lu", ( long unsigned int) eod );
      active_times[ idx++ ] = str_dup( buffer, len );
   }

   return active_times;
}

char *get_eod( char *t, char **day )
{
   time_t fod, eod;
   struct tm *tm;

   fod = ( time_t )strtol( t, NULL, 10 );
   eod = 0;

   *day = NULL;

   if (( tm = localtime( &fod )) == NULL )
   {
      syslog( LOG_ERR, "get_eod(): localtime( &fod ) failed\n" );
      return NULL;
   }

   snprintf( day_buffer, sizeof( day_buffer ),
      "%d %s <a href=\"javascript:ephemera.load('request=calendar&year=%d')\">%d</a>",
      tm->tm_mday, months[ tm->tm_mon ], tm->tm_year + 1900, tm->tm_year + 1900 );

   *day = day_buffer;

   tm->tm_hour = 23;
   tm->tm_min = 59;
   tm->tm_sec = 59;
   tm->tm_isdst = -1;

   if (( eod = mktime( tm )) < 0 )
   {
      syslog( LOG_ERR, "get_eod(): mktime() returned an error" );
      return NULL;
   }

   snprintf( eod_buffer, sizeof( eod_buffer ), "%lu", ( long unsigned int)eod );
   return eod_buffer;
}

char *get_eom( char *t, char **month )
{
   time_t fom, eom;
   struct tm *tm;

   *month = NULL;

   fom = ( time_t )strtol( t, NULL, 10 );
   eom = 0;

   if (( tm = localtime( &fom )) == NULL )
   {
      syslog( LOG_ERR, "get_eom(): localtime( &fom ) failed\n" );
      return NULL;
   }

   snprintf( month_buffer, sizeof( month_buffer ),
      "%s <a href=\"javascript:ephemera.load('request=calendar&year=%d')\">%d</a>",
      months[ tm->tm_mon ], tm->tm_year + 1900, tm->tm_year + 1900 );

   *month = month_buffer;

   if ( tm->tm_mon == 11 )
   {
      ++tm->tm_year;
      tm->tm_mon = 0;
   }
   else
      ++tm->tm_mon;

   tm->tm_mday = 1;
   tm->tm_hour = 0;
   tm->tm_min = 0;
   tm->tm_sec = 0;
   tm->tm_isdst = -1;

   if (( eom = mktime( tm )) < 0 )
   {
      syslog( LOG_ERR, "get_eom(): mktime() returned an error" );
      return NULL;
   }

   snprintf( eom_buffer, sizeof( eom_buffer ), "%lu", ( long unsigned int)( eom - 1 ));
   return eom_buffer;
}

char *markup_calendar( int year, int mon, int first, int last, int num_active, char **active, char *t )
{
   static char *empty = "<td>&nbsp;</td>";
   static struct string *str = NULL;
   int idx, ord, day, tmp, row;
   char *ptr, buffer[ 1024 ], hbuffer[ 1024 ];

   if ( str == NULL )
      str = make_string();
   else
   {
      str->free += str->used;
      str->used = 0;
      str->top = str->str;
   }

   if ( num_active )
      snprintf( hbuffer, sizeof( hbuffer ), "<b><a href=\"javascript:ephemera.load('request=month&amp;time=%s')\">%s %d</a></b>",
                t, months[ mon - 1 ], year );
   else
      snprintf( hbuffer, sizeof( hbuffer ), "<b>%s %d</b>", months[ mon - 1 ], year );

   snprintf( buffer, sizeof( buffer ),
             "<div class=\"mondiv\">\n"
             "<table class=\"calendar\">\n"
             "<thead>\n"
             "<tr><td colspan=\"7\">%s</td></tr>\n"
             "<tr><td>Sun</td><td>Mon</td><td>Tue</td><td>Wed</td><td>Thu</td><td>Fri</td><td>Sat</td></tr>\n"
             "</thead>\n"
             "<tbody>\n"
             "<tr>",

             hbuffer );

   for( ptr = buffer; *ptr; ++ptr )
      STRING_APPEND( str, *ptr )

   tmp = weekday( first );
   tmp = ( tmp == 6 ? 0 : tmp % 6 + 1 );

   if ( tmp > 0 )
      tmp = first - tmp;
   else
      tmp = first;

   for( row = 1, idx = 1, ord = 1, day = tmp; day < last; ++day, ++idx )
   {
      if ( day < first )
         ptr = empty;
      else
      {
         if ( active[ ord - 1 ] != NULL )
            snprintf( buffer, sizeof( buffer ),
                      "<td class=\"active\"><a href=\"javascript:ephemera.load('request=day&amp;time=%s')\">%d</a></td>",
                      active[ ord - 1 ], ord );
         else
            snprintf( buffer, sizeof( buffer ), "<td>%d</td>", ord );

         ptr = buffer;
         ++ord;
      }

      for( ; *ptr; ++ptr )
         STRING_APPEND( str, *ptr )

      if ( !( idx % 7 ))
      {
         ++row;

         for( ptr = "</tr>\n<tr>"; *ptr; ++ptr )
            STRING_APPEND( str, *ptr )
      }
   }

   --idx;

   if ( !( idx % 7 ))
   {
      str->used -= 4;
      str->free += 4;
      str->top -= 4;
      *str->top = '\0';

      --row;
   }
   else
   {
      while( idx % 7 )
      {
         for( ptr = empty; *ptr; ++ptr )
            STRING_APPEND( str, *ptr )

         ++idx;
      }

      for( ptr = "</tr>\n"; *ptr; ++ptr )
         STRING_APPEND( str, *ptr )
   }

   while( row < 6 )
   {
      for( ptr = "<tr>"; *ptr; ++ptr )
         STRING_APPEND( str, *ptr )

      for( tmp = 0; tmp < 7; ++tmp )
         for( ptr = empty; *ptr; ++ptr )
            STRING_APPEND( str, *ptr )

      for( ptr = "</tr>\n"; *ptr; ++ptr )
         STRING_APPEND( str, *ptr )

      ++row;
   }

   for( ptr = "</tbody>\n</table>\n</div>\n"; *ptr; ++ptr )
      STRING_APPEND( str, *ptr )

   return str->str;
}

char *make_calendar( int year, int mon, int num_active, char **active )
{
   struct date date;
   int first, last;
   struct tm tm;
   time_t t;
   char tbuffer[ 64 ];

   if ( ! num_active )
      *tbuffer = '\0';
   else
   {
      tm.tm_year = year - 1900;
      tm.tm_mon = mon - 1;
      tm.tm_mday = 1;
      tm.tm_hour = 0;
      tm.tm_min = 0;
      tm.tm_sec = 0;
      tm.tm_isdst = -1;

      if (( t = mktime( &tm )) < 0 )
      {
         syslog( LOG_ERR, "make_calendar(): mktime( &tm ) failed\n" );
         return NULL;
      }

      snprintf( tbuffer, sizeof( tbuffer ), "%lu", ( long unsigned int)t );
   }

   date.y = year;
   date.m = mon;
   date.d = 1;

   if (( first = ndaysg( &date )) < 0 )
   {
      syslog( LOG_ERR, "make_calendar(): ndaysg() returned an error" );
      return NULL;
   }

   if ( date.m < 12 )
      ++date.m;
   else
   {
      ++date.y;
      date.m = 1;
   }

   if (( last = ndaysg( &date )) < 0 )
   {
      syslog( LOG_ERR, "make_calendar(): ndaysg() returned an error" );
      return NULL;
   }

   return markup_calendar( year, mon, first, last, num_active, active, tbuffer );
}
