/*
 * Ephemera Copyright (c) 2014, 2018, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <syslog.h>
#include <time.h>
#include <stdlib.h>
#include <sys/malloc.h>
#include <sqlite3.h>

#include "data.h"

int db_busy_handler( void *data, int tries )
{
   struct timespec tv;

   if ( tries == 1000000 )
      return 0;

   tv.tv_sec = 0;
   tv.tv_nsec = 10000;

   nanosleep( &tv, NULL );

   return 1;
}

sqlite3 *sqlite_open( char *filename )
{
   sqlite3 *db;
   const char *sql_err = NULL;

   if( sqlite3_open( filename, &db ) != SQLITE_OK )
   {
      sql_err = sqlite3_errmsg( db );
      syslog( LOG_ERR, "sqlite3_open( %s ): %s", filename, ( char *)sql_err );
      sqlite3_close( db );
      return NULL;
   }

   sqlite3_busy_handler( db, db_busy_handler, NULL );

   return db;
}

int sqlite_exec( sqlite3 *db, char *sql )
{
   char *sql_err;
      
   if ( sqlite3_exec( db, sql, NULL, NULL, &sql_err ) != SQLITE_OK )
   {
      syslog( LOG_ERR, "sqlite3_exec( %s ): %s", ( sql == NULL ? "NULL" : sql ), sql_err );
      free( sql_err );
      return -1;
   }

   return 0;
}

sqlite3_stmt *sqlite_compile( sqlite3 *db, char *sql, int len )
{
   sqlite3_stmt *compiled;
   const char *ignored;

   if ( sqlite3_prepare_v2( db, sql, len, &compiled, &ignored ) != SQLITE_OK )
   {
      syslog( LOG_ERR, "sqlite3_prepare_v2(): %s", ( char *)sqlite3_errmsg( db ));
      return NULL;
   }

   return compiled;
}

int sqlite_bind( sqlite3_stmt *sql, int pos, char *item, int len )
{
   if ( sqlite3_bind_text( sql, pos, item, len, SQLITE_TRANSIENT ) != SQLITE_OK )
   {
      syslog( LOG_ERR, "sqlite3_bind_text:(): %s", ( char *)sqlite3_errmsg( sqlite3_db_handle( sql )));
      return -1;
   }

   return 0;
}

int sqlite_step( sqlite3_stmt *sql )
{
   int code;

   if (( code = sqlite3_step( sql )) != SQLITE_ROW )
   {
      if ( code == SQLITE_DONE )
         return 0;
      else
      {
         syslog( LOG_ERR, "sqlite3_step(): %s", ( char *)sqlite3_errmsg( sqlite3_db_handle( sql )));
         return -1;
      }
   }

   return 1;
}

int sqlite_row( sqlite3_stmt *sql, struct stack **stk )
{
   int total, i, len;
   char *column = NULL;

   total = sqlite3_column_count( sql );

   for( i = 0; i < total; ++i )
   {
      if ( ! ( len = sqlite3_column_bytes( sql, i )))
         STACK_PUSH( *stk, str_dup( "", 0 ))
      else if (( column = ( char *)sqlite3_column_text( sql, i )) == NULL )
      {
         syslog( LOG_ERR, "sqlite3_column_text(): %s", ( char *)sqlite3_errmsg( sqlite3_db_handle( sql )));
         goto ERR;
      }
      else
         STACK_PUSH( *stk, str_dup( column, -1 ))
   }

   return 0;

ERR:
   while(( *stk )->used )
      free( stack_pop( *stk ));

   return -1;
}

int sqlite_reset( sqlite3_stmt *sql )
{
   if ( sqlite3_reset( sql ) != SQLITE_OK )
   {
      syslog( LOG_ERR, "sqlite3_reset(): %s", ( char *)sqlite3_errmsg( sqlite3_db_handle( sql )));
      return -1;
   }

   return 0;
}

void sqlite_free( sqlite3_stmt *sql )
{
   if ( sqlite3_finalize( sql ) != SQLITE_OK )
      syslog( LOG_ERR, "sqlite_finalize(): %s", ( char *)sqlite3_errmsg( sqlite3_db_handle( sql )));
}
