 /*
 * Avalon Copyright (c) 2020, 2022, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <term.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/event.h>

#include "messageclient.h"

unsigned char *password = ( unsigned char *)PASSWORD;
char *af = NULL, *ab = NULL, *port = "10000", *transcript = NULL;

FILE *t_file = NULL;
int ignore = 0;

void green_fore()
{
   if ( af == NULL )
      return;

   putp( tparm( ab, COLOR_BLACK ));
   putp( tparm( af, COLOR_GREEN ));
}

void white()
{
   if ( af == NULL )
      return;

   putp( tparm( ab, COLOR_BLACK ));
   putp( tparm( af, COLOR_WHITE ));
}

void red()
{
   if ( af == NULL )
      return;

   putp( tparm( ab, COLOR_WHITE ));
   putp( tparm( af, COLOR_RED ));
}

void show_message( char *msg )
{
   fputs( msg, stdout );
   fputc( '\n', stdout );
   fflush( stdout );
}

void show_error()
{
   red();
   printf( "%s\n", mc_error );
   white();
   fflush( stdout );
}

void show_io_error( char *msg )
{
   fprintf( stderr, "%s: %s\n", msg, strerror( errno ));
   fflush( stderr );
}

/*
 * We know that when we receive a read event for the socket, a complete
 * message will be delivered.  If we block, it will be only until the
 * all the message has arrived.
 */

int read_message( struct mc_ssl *conn )
{
   unsigned char *response;
   int len;

   if (( len = mc_read_message( conn, &response )) <= 0 )
   {
      if ( len == -1 )
      {
         show_error();
         return 1;
      }

      return len;
   }

   if ( len && *response == '-' )
      green_fore();

   if ( len && ! fwrite( response, len, 1, stdout ))
      show_io_error( "could not write to terminal" );

   if ( len && *response == '-' )
      white();

   fflush( stdout );

   if ( t_file != NULL && len )
   {
      if ( fwrite( response, len, 1, t_file ))
         fflush( t_file );
      else
      {
         show_io_error( "could not write to transcript" );
         fclose( t_file );
         t_file = NULL;
      }
   }

   free( response );
   return 0;
}

/*
 * We leave the console in canonical mode, which means that read events
 * will be generated when the user presses [Enter].  We know, therefore,
 * that we can read without blocking until we encounter the line feed.
 */

int read_line( unsigned char *buffer, int size )
{
   unsigned char c, *ptr;
   int r, count;

   count = 0;
   ptr = buffer;

   for( ; ; )
   {
      if ( !( r = fread( &c, 1, 1, stdin )))
      {
         if ( ferror( stdin ))
         {
            show_io_error( "could not read terminal input" );
            return r;
         }

         return count;
      }

      ++count;
      *ptr++ = c;
      *ptr = '\0';

      if ( c == 10 || count == size - 1 )
         break;
   }

   return count;
}

/*
 * The user needs to lock the screen so that it does not scroll when the
 * user types.  Pressing [Enter] on a blank line locks and unlocks the
 * screen.

 * The user does not need to lock the screen for the first line typed.  The
 * first line contains the user's handle.  The server will not deliver data
 * for a connection until the server receives the handle.

 * We send out each line the user types as a fragment frame.  When the
 * user unlocks the screen, we send an empty terminator frame.  Any line
 * typed on an unlocked screen is discarded.
 */

int send_message( struct mc_ssl *conn )
{
   static int first = 1;
   int len, lock = 0;
   unsigned char buffer[ 1024 ];

AGAIN:
   if ( !( len = read_line( buffer, sizeof( buffer ))))
   {
      show_message( "[ Disconnecting. ]" );
      return 1;
   }

   if ( first )
   {
      if (( len = mc_send_message( conn, len, buffer )))
         goto ERROR;

      --first;
      return 0;
   }

   if ( len == 1 )
   {
      if ( *buffer == '\n' )
      {
         if (( lock ^= 1 ))
         {
            show_message( "[ Locked ]" );
            goto AGAIN;
         }

         show_message( "[ Unlocked ]" );

         if (( len = mc_send_frame( conn, 1, 0, buffer )))
            goto ERROR;

         return 0;
      }
   }
   else if ( len == 2 && ! lock )
   {
      if ( *buffer == '.' || *buffer == '?' )
      {
         /*
          * Length must be 1 for server to recognize command.
          */

         if (( len = mc_send_message( conn, 1, buffer )))
            goto ERROR;

         return 0;
      }

      if ( *buffer == 't' )
      {
         if ( t_file == NULL )
            show_message( "[ No transcript file is open. ]" );
         else
         {
            show_message( "[ Starting pager. ]" );
            snprintf(( char * )buffer, sizeof( buffer ), "more %s", transcript );
            system(( char * )buffer );
            show_message("[ Pager finished. ]" );
         }

         return 0;
      }
   }

   if ( lock )
   {
      if (( len = mc_send_frame( conn, 0, len, buffer )))
         goto ERROR;

      goto AGAIN;
   }

   show_message( "[ Discarded. Lock the screen to type a message. Unlock to send it. ]" );
   return 0;

ERROR:
   if ( len == -1 )
      show_error();

   return 1;
}

/*
 * We must be event-driven because both halves of the connection can write
 * at any time.

 * We do not need to use non-blocking descriptors because both client and
 * server deliver known structure.  The client types lines of
 * terminated text.  The server delivers complete messages.
 */

void process_streams( struct mc_ssl *conn )
{
   int i, n, r, kq;
   struct kevent inq[ 2 ], outq[ 2 ];

   inq[ 0 ].ident  = STDIN_FILENO;
   inq[ 0 ].filter = EVFILT_READ;
   inq[ 0 ].fflags = 0;
   inq[ 0 ].flags  = EV_ADD | EV_ENABLE;

   inq[ 1 ].ident  = conn->fd;
   inq[ 1 ].filter = EVFILT_READ;
   inq[ 1 ].fflags = 0;
   inq[ 1 ].flags  = EV_ADD | EV_ENABLE;

   if (( kq = kqueue()) < 0 )
   {
      show_io_error( "kqueue() failed" );
      return;
   }

   if ( kevent( kq, ( struct kevent *)&inq, 2, NULL, 0, NULL ) < 0 )
   {
      show_io_error( "kevent() failed" );
      return;
   }

   for( ; ; )
   {
      if (( n = kevent( kq, NULL, 0, ( struct kevent *)&outq, 2, NULL )) <= 0 )
         break;

      for( i = 0; i < n; ++i )
      {
         if ( outq[ i ].flags & EV_ERROR )
            break;

         if ( outq[ i ].ident == STDIN_FILENO )
            r = send_message( conn );
         else
            r = read_message( conn );

         if ( r )
            goto ESCAPE;
      }
   }

ESCAPE:
   close( kq );
}

void authorize( struct mc_ssl *conn )
{
   unsigned int len;
   unsigned char *ptr;

   len = 0;

   for( ptr = password; *ptr; ++ptr )
      ++len;

   if (( len = mc_send_message( conn, len, password )))
   {
      if ( len == -1 )
         show_error();

      exit( 1 );
   }
}

int process_args( int argc, char **argv )
{
   int i, n = 0;

   while(( i = getopt( argc, argv, "ip:t:" )) != -1 )
   {
      ++n;

      switch( i )
      {
         case '?':
            exit( 1 );

         case 'i':
            ++ignore;
            break;

         case 'p':
            port = optarg;
            ++n;
            break;

        case 't':
            transcript = optarg;
            ++n;
      }
   }

   if ( transcript != NULL && ( t_file = fopen( transcript, "w" )) == NULL )
      show_io_error( "could not open transcript file" );

   return n;
}

void setup_colors()
{
   if ( setupterm( NULL, 1, NULL ) != OK )
      return;

   if (( long int )( af = tigetstr( "setaf" )) <= 0 || ( long int )( ab = tigetstr( "setab" )) <= 0 )
   {
      af = NULL;
      ab = NULL;
   }
}

void show_intro( char *host )
{
   char *strings[] = { "[ Avalon Client " VERSION " ]", "[ [Enter] locks/unlocks the screen. ]",
                        "[ Control-D disconnects. ]", NULL };
   char buffer[ 128 ];
   char **ptr;

   for( ptr = strings; *ptr != NULL; ++ptr )
      show_message( *ptr );

   snprintf( buffer, sizeof( buffer ), "[ Connecting to %s on port %s ]", host, port );
   show_message( buffer );

   if ( t_file != NULL )
      show_message( "[ t [Enter] displays the transcript. ]" );
}

int main( int argc, char **argv )
{
   int n;
   struct mc_ssl *conn;

   setup_colors();

   n = process_args( argc, argv );
   argc -= n;
   argv += n;

   if ( argc < 2 )
   {
      show_message( "usage: aclient [-i] [-p <port>] [-t <filename>] <hostname|address>\n"
                    "usage: -i -- Ignore mismatch of server hostname with TLS certificate hostname.\n"
                    "usage: -p -- Connect to <port>.  Default is 10000.\n"
                    "usage: -t -- Write transcript of conversation to <filename>.\n" );
      exit( 1 );
   }

   if ( mc_init() )
      exit( 1 );

   show_intro( argv[ 1 ] );

   if (( conn = mc_connect_to_server( argv[ 1 ], port, ignore )) == NULL )
   {
      show_error();
      exit( 1 );
   }

   authorize( conn );
   process_streams( conn );
   mc_close_connection( conn );
   mc_deinit();

   if ( t_file != NULL )
      fclose( t_file );

   return 0;
}
