/*
 * LibMessage Server Library Copyright (c) 2018-2025 James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <sys/wait.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

#include <pwd.h>
#include <grp.h>

#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/dh.h>
#include <openssl/ec.h>
#include <openssl/bio.h>
#include <openssl/x509.h>
#include <openssl/crypto.h>
#include <openssl/tls1.h>

#define MAX_MESSAGE 262128
#define MAX_FRAME 32766
#define TERMINATOR 32768

#define QLEN 16384
#define MAX_EVENT 10

SSL_CTX *ctx;

struct kevent *ms_inqueue, *ms_outqueue;

extern char *optarg;

int ms_idx = 0, ms_in = 0, ms_out = 0, ms_fd = -1, ms_logging = 0, ms_testing = 0,
    ms_timer = 0, ms_max_conn = QLEN, ms_qlen = QLEN * MAX_EVENT;

unsigned int ms_max_message = MAX_MESSAGE;

unsigned int ms_stack_inc = 128, ms_backlog = 1000, ms_conn_count = 0;

pid_t ms_worker_pid = 0;

char *ms_grp = "nobody", *ms_user = "nobody", *ms_config_file = NULL,
     *ms_interface = "", *ms_port = "10000", *ms_tls_file = NULL, *ms_rootdir = NULL,
     *ms_keyfile = NULL, *ms_chainfile = NULL, *ms_password = NULL,
     *ms_pidfile = "/var/run/message.pid", *ms_app_name = "message";

void ( *ms_periodic )() = NULL;

struct passwd *ms_passwd;
struct group *ms_group;

volatile int ms_sigterm = 0;

/*
 * Queue of buffered out-going data for each connection.
 */

struct ms_buffer
{
   unsigned int total, refcount;
   char *buffer;
};

struct ms_queue
{
   struct ms_buffer *buffer;
   struct ms_queue *next;
   ssize_t offset, len;
   int sendfile, finished;
};

/*
 * Connection control block.
 */

struct ms_ccb
{
   SSL *ssl;
   int sock;
   unsigned int len, offset, qlen, opposite;
   unsigned char type, stage, closed, *buffer;
   void *data;
   struct ms_queue *first, *last;
};

void ms_init_func();
void ms_exit_func();
int ms_accept_callback( void * );
int ms_open_callback( void * );
void ms_close_callback( void * );
int ms_read_callback( void *, unsigned int, unsigned char * );
void ms_write_callback( void * );

void ms_set_periodic( void (*)(), int );
int ms_queue_message( struct ms_ccb **, int, unsigned int, unsigned char * );
void ms_close_conn( struct ms_ccb *, int );

void ms_set_events( struct ms_ccb *, int, int );

void ms_log_error()
{
   unsigned long err;
   char buffer[ 128 ];

   while(( err = ERR_get_error()))
   {
      *buffer = '\0';
      ERR_error_string_n( err, buffer, sizeof( buffer ));

      if ( ms_logging )
         syslog( LOG_ERR, "%s: OpenSSL: %s", ms_app_name, buffer );
      else
         fprintf( stderr, "%s: OpenSSL: %s\n", ms_app_name, buffer );
   }
}

void *ms_memory( size_t size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
   {
      if ( ms_logging )
         syslog( LOG_WARNING, "malloc(): %m" );
      else
         fprintf( stderr, "%s: malloc(): %s\n", ms_app_name, strerror( errno ));

      return NULL;
   }

   return ptr;
}

char *ms_str_dup( char *str )
{
   char *ptr;
   int len;

   for( len = 0, ptr = str; *ptr; ++ptr )
      ++len;

   if (( ptr = ms_memory( len + 1 )) != NULL )
   {
      bcopy( str, ptr, len );
      ptr[ len ] = '\0';
   }

   return ptr;
}

void ms_ev_set( int desc, short filter, u_short flags, struct ms_ccb *item )
{
   struct kevent *kev;

   if ( ms_in >= ms_qlen )
      return;

   kev = &ms_inqueue[ ms_in++ ];

   kev->ident = desc;
   kev->filter = filter;
   kev->fflags = 0;
   kev->flags = flags;
   kev->udata = item;
}

void ms_stop_read_socket( struct ms_ccb *item )
{
   ms_ev_set( item->sock, EVFILT_READ, EV_DISABLE, item );
}

void ms_start_read_socket( struct ms_ccb *item )
{
   ms_ev_set( item->sock, EVFILT_READ, EV_ENABLE, item );
}

void ms_stop_write_socket( struct ms_ccb *item )
{
   ms_ev_set( item->sock, EVFILT_WRITE, EV_DISABLE, item );
}

void ms_start_write_socket( struct ms_ccb *item )
{
   ms_ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
}

void ms_set_timer()
{
   struct kevent *kev;

   if ( ms_in >= ms_qlen )
      return;

   kev = &ms_inqueue[ ms_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = ms_timer * 1000;
   kev->flags = EV_ADD;
   kev->udata = NULL;
}

void ms_delete_timer()
{
   struct kevent *kev;

   if ( ms_in >= ms_qlen )
      return;

   kev = &ms_inqueue[ ms_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = 0;
   kev->flags = EV_DELETE;
   kev->udata = NULL;
}

void ms_set_periodic( void ( *callback )(), int seconds )
{
   if (( ms_timer = seconds ) <= 0 || callback == NULL )
      ms_timer = 0;

   ms_delete_timer();

   if ( ms_timer )
      ms_set_timer();

   ms_periodic = callback;
}

void ms_non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }
}

void ms_set_options( int argc, char **argv )
{
   int i;

   while(( i = getopt( argc, argv, "xr:s:f:i:p:m:n:u:g:t:" )) != -1 )
      switch( i )
      {
         case 'f':
            ms_config_file = optarg;
            break;

         case 'g':
            ms_grp = optarg;
            break;

         case 'i':
            ms_interface = optarg;
            break;

         case 'm':
            ms_max_conn = strtol( optarg, NULL, 10 );
            break;

         case 'p':
            ms_port = optarg;
            break;

         case 'r':
            ms_rootdir = optarg;
            break;

         case 's':
            ms_max_message = strtol( optarg, NULL, 10 );
            break;

         case 'u':
            ms_user = optarg;
            break;

         case 't':
            ms_tls_file = optarg;
            break;

         case 'x':
            ms_testing = 1;
      }

   if ( ms_tls_file == NULL )
   {
      fprintf( stderr, "%s: -t option missing\n", ms_app_name );
      exit( 1 );
   }

   if ( ms_rootdir == NULL )
   {
      fprintf( stderr, "%s: -r option missing\n", ms_app_name );
      exit( 1 );
   }

   if ( ms_max_message % MAX_FRAME )
   {
      fprintf( stderr, "%s: -s <max-message> must be a multiple of the frame size of %d .", ms_app_name, MAX_FRAME );
      exit( 1 );
   }

   if ( ms_max_conn <= 0 )
   {
      fprintf( stderr, "%s: -m <max-conn> <= 0: %d\n", ms_app_name, ms_max_conn );
      exit( 1 );
   }

   if (( ms_passwd = getpwnam( ms_user )) == NULL )
   {
      fprintf( stderr, "%s: user \"%s\" does not exist\n", ms_app_name, ms_user );
      exit( 1 );
   }

   if (( ms_group = getgrnam( ms_grp )) == NULL )
   {
      fprintf( stderr, "%s: group \"%s\" does not exist\n", ms_app_name, ms_grp );
      exit( 1 );
   }
}

void ms_master_sigterm_handler( int signo )
{
   if ( ms_worker_pid <= 0 )
      return;

   kill( ms_worker_pid, signo );
}

void ms_set_signals()
{
   int *iptr;
   int sigs[] = {
      SIGPIPE, SIGHUP, SIGQUIT, SIGUSR1, SIGUSR2, SIGALRM, SIGINT, SIGTSTP, -1
   };

   signal( SIGTERM, ms_master_sigterm_handler );

   for( iptr = sigs; *iptr > 0; ++iptr )
      signal( *iptr, SIG_IGN );
}

/*
 * fork() child to handle connections.  Parent waits for the child to exit.
 * If the child is killed by a signal, parent spawns another.  If the child
 * exit()s, then the parent exits.  The child returns from become_daemon().
 * The parent does not.  Parent installs a new signal handler for SIGTERM.
 * Upon receipt, the handler propagates the signal to child.
 */

void ms_fork_master_worker()
{
   int status;

AGAIN:
   switch(( ms_worker_pid = fork() ))
   {
      case -1:
         syslog( LOG_ERR, "fork(): %m" );
         exit( 1 );

      case 0:
         break;

      default:
         status = 0;

         if ( waitpid( ms_worker_pid, &status, WEXITED ) < 0 )
         {
            syslog( LOG_ERR, "waitpid(): %m" );
            exit( 1 );
         }

         if ( WIFSIGNALED( status ))
            goto AGAIN;

         exit( 0 );
   }
}

void ms_become_daemon( char **argv )
{
   char buffer[ 16 ];
   int t, file;

   /*
    * Fork and let the parent die, continuing as child so we are not
    * a process group leader.  This is necessary for the call to setsid().
    */

   switch( fork() )
   {
      case -1:
         fprintf( stderr, "%s: fork(): %s\n", ms_app_name, strerror( errno ));
         exit( 1 );

      case 0:
         break;

      default:
         exit( 0 );
   }

   fclose( stdout );
   fclose( stderr );
   fclose( stdin );

   stdin = fopen( "/dev/null", "r" );
   stdout = fopen( "/dev/null", "w" );
   stderr = fopen( "/dev/null", "w" );

   /*
    * We inherit the syslog connection.
    */

   if ( stdin == NULL || stdout == NULL || stderr == NULL )
   {
      syslog( LOG_ERR, "fopen(): %m" );
      exit( 1 );
   }

   /*
    * Detach us from our controlling terminal, so job control and other
    * signals may not be sent to us from that terminal.
    */

   if ( setsid() < 0 )
   {
      syslog( LOG_ERR, "setsid(): %m" );
      exit( 1 );
   }

   /*
    * We don't create any files or fork children which create files,
    * but if we did, this will make those files not world-readable by
    * default.
    */

   umask( 2 );

   /*
    * Write our pid to disk.
    */

   if (( file = open( ms_pidfile, O_WRONLY | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP )) < 0 )
   {
      syslog( LOG_WARNING, "open( PIDFILE ): %m" );
      return;
   }

   t = snprintf( buffer, sizeof( buffer ), "%d", getpid() );
   write( file, buffer, t );
   close( file );

   ms_fork_master_worker();
}

void ms_change_identity()
{
   if ( setgid( ms_group->gr_gid ) < 0 )
      syslog( LOG_ERR, "setgid(): %m" );

   if ( setuid( ms_passwd->pw_uid ) < 0 )
      syslog( LOG_ERR, "setuid(): %m" );
}

void ms_close_on_exec( int fd )
{
   if ( fcntl( fd, F_SETFD, FD_CLOEXEC ) < 0 )
      syslog( LOG_ERR, "fcntl( F_SETFD, FD_CLOEXEC ): %m" );
}

void ms_start_listening()
{
   struct addrinfo hints, *res;
   int result;

   /*
    * This is the new, protocol-independent way of setting up a listening
    * socket.
    */

   bzero( &hints, sizeof( struct addrinfo ));
   hints.ai_flags = AI_PASSIVE;
   hints.ai_socktype = SOCK_STREAM;

   /*
    * If the user has not specified an interface, we listen on the
    * IPv6 wildcard address.
    */

   if (( result = getaddrinfo(( *ms_interface ? ms_interface : NULL ), ms_port, &hints, &res )))
   {
      syslog( LOG_ERR, "getaddrinfo(): %s", gai_strerror( result ));
      exit( 1 );
   }

   if ( res == NULL )
   {
      syslog( LOG_ERR, "getaddrinfo(): no interface found" );
      exit( 1 );
   }

   ms_fd = socket( res->ai_family, res->ai_socktype, res->ai_protocol );

   if ( ms_fd == -1 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   result = 1;

   /*
    * Allow duplicate bindings, so we can have more than one
    * server process, if necessary.
    */

   if ( setsockopt( ms_fd, SOL_SOCKET, SO_REUSEPORT, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_REUSEPORT ): %m" );

   /*
    * Try and detect connections which go idle for long periods of time.
    */

   if ( setsockopt( ms_fd, SOL_SOCKET, SO_KEEPALIVE, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_KEEPALIVE ): %m" );

   /*
    * Make sure we can accept IPv4 traffic as well as IPv6, if we
    * are bound to the IPv6 wildcard address.  We are turning off
    * the option because result = 0.
    */

   result = 0;

   if ( ! *ms_interface && setsockopt( ms_fd, IPPROTO_IPV6, IPV6_BINDV6ONLY, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( IPV6_BINDV6ONLY ): %m" );

   if ( bind( ms_fd, res->ai_addr, res->ai_addrlen ) < 0 )
   {
      syslog( LOG_ERR, "bind(): %m" );
      exit( 1 );
   }

   if ( res != NULL )
      freeaddrinfo( res );

   if ( listen( ms_fd, ms_backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      exit( 1 );
   }

   ms_close_on_exec( ms_fd );
   ms_non_blocking( ms_fd );
}

void ms_set_name( char *name )
{
   char *ptr1, *ptr2;
   int len;

   if ( name == NULL )
      return;

   for( len = 0, ptr1 = name; *ptr1; ++ptr1 )
      ++len;

   ms_app_name = name;

   if (( ptr2 = ms_pidfile = ms_memory( 14 + len )) == NULL )
      exit( 1 );

   for( ptr1 = "/var/run/"; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = name; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = ".pid"; *ptr1; )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';
}

void ms_set_data( struct ms_ccb *item, void *data )
{
   if ( item != NULL )
      item->data = data;
}

void *ms_get_data( struct ms_ccb *item )
{
   if ( item != NULL )
      return item->data;

   return NULL;
}

unsigned int ms_get_qlen( struct ms_ccb *item )
{
   if ( item != NULL )
      return item->qlen;

   return 0;
}

int ms_get_max_conn()
{
	return ms_max_conn;
}

unsigned int ms_get_max_message()
{
	return ms_max_message;
}

unsigned int ms_set_max_message( unsigned int max )
{
   if ( ! max || ( max % MAX_FRAME ))
      return 0;

   return ms_max_message = max;
}

int ms_queue_each_client( struct ms_ccb **items, int conns, struct ms_buffer *buffer, int fd, ssize_t offset, ssize_t len )
{
   struct ms_ccb **ptr;
   struct ms_queue *q;
   int n;

   for( n = 0, ptr = items; n < conns; ++n, ++ptr )
   {
      if (( *ptr )->closed )
         continue;

      if (( q = ms_memory( sizeof( struct ms_queue ))) == NULL )
         return 1;

      q->finished = 0;
      q->sendfile = fd;
      q->offset   = offset;
      q->len      = len;
      q->buffer   = buffer;

      ++q->buffer->refcount;
      ++( *ptr )->qlen;

      if (( *ptr )->first == NULL )
      {
         ( *ptr )->first = ( *ptr )->last = q;
         ms_ev_set(( *ptr )->sock, EVFILT_WRITE, EV_ENABLE, *ptr );
      }
      else
      {
         ( *ptr )->last->next = q;
         ( *ptr )->last = q;
      }

      ( *ptr )->last->next = NULL;
   }

   return 0;
}

int ms_queue_frame( struct ms_ccb **items, int conns, int final, unsigned int len, unsigned char *data )
{
   unsigned int code, header;
   struct ms_buffer *buffer;

   if ( ! conns )
      return 0;

   if ( len > MAX_FRAME )
      return 1;

   if (( buffer = ms_memory( sizeof( struct ms_buffer ))) == NULL )
      return 1;

   if (( buffer->buffer = ms_memory( len + 2 )) == NULL )
   {
      free( buffer );
      return 1;
   }

   code   = ( final ? TERMINATOR : 0 );
   header = code + len;

   buffer->buffer[ 0 ] = header / 256;
   buffer->buffer[ 1 ] = header % 256;

   buffer->total    = len + 2;
   buffer->refcount = 0;

   bcopy( data, &buffer->buffer[ 2 ], len );

   if ( ms_queue_each_client( items, conns, buffer, 0, 0, 0 ))
   {
      if ( ! buffer->refcount )
      {
         free( buffer->buffer );
         free( buffer );
      }

      return 1;
   }

   return 0;
}

int ms_queue_message( struct ms_ccb **items, int conns, unsigned int len, unsigned char *data )
{
   unsigned int frames, code, flen, total, i, header;
   unsigned char *payload;
   struct ms_buffer *buffer;

   if ( ! conns )
      return 0;

   if ( len > ms_max_message )
      return 1;

   if ( ! ( frames = len / MAX_FRAME ))
      ++frames;
   else if ( len % MAX_FRAME )
      ++frames;

   total   = len;
   payload = data;

   for( i = 0; i < frames; ++i )
   {
      if (( buffer = ms_memory( sizeof( struct ms_buffer ))) == NULL )
         return 1;

      if ( i == frames - 1 )
      {
         flen = total;
         code = TERMINATOR;
      }
      else
      {
         flen = MAX_FRAME;
         code = 0;
      }

      if (( buffer->buffer = ms_memory( flen + 2 )) == NULL )
      {
         free( buffer );
         return 1;
      }

      header = code + flen;

      buffer->buffer[ 0 ] = header / 256;
      buffer->buffer[ 1 ] = header % 256;

      buffer->total    = flen + 2;
      buffer->refcount = 0;

      bcopy( payload, &buffer->buffer[ 2 ], flen );

      total   -= MAX_FRAME;
      payload += MAX_FRAME;

      if ( ms_queue_each_client( items, conns, buffer, 0, 0, 0 ))
      {
         if ( ! buffer->refcount )
         {
            free( buffer->buffer );
            free( buffer );
         }

         return 1;
      }
   }

   return 0;
}

int ms_queue_sendfile_message( struct ms_ccb **items, int conns, int fd, ssize_t offset, ssize_t len )
{
   unsigned int frames, code, flen, total, i, header;
   ssize_t off;
   struct ms_ccb **ptr;
   struct ms_buffer *buffer;

   if ( ! conns )
      return 0;

   if ( len > ms_max_message )
      return 1;

   for( i = 0, ptr = items; i < conns; ++i, ++ptr )
   {
      BIO *bio;

      if (( *ptr )->closed )
         continue;

      if (( bio = SSL_get_wbio(( *ptr )->ssl )) != NULL && ! BIO_get_ktls_send( bio ))
         return 2;
   }

   if ( ! ( frames = len / MAX_FRAME ))
      ++frames;
   else if ( len % MAX_FRAME )
      ++frames;

   total = len;
   off   = offset;

   for( i = 0; i < frames; ++i )
   {
      if (( buffer = ms_memory( sizeof( struct ms_buffer ))) == NULL )
         return 1;

      if ( i == frames - 1 )
      {
         flen = total;
         code = TERMINATOR;
      }
      else
      {
         flen = MAX_FRAME;
         code = 0;
      }

      if (( buffer->buffer = ms_memory( 2 )) == NULL )
      {
         free( buffer );
         return 1;
      }

      header = code + flen;

      buffer->buffer[ 0 ] = header / 256;
      buffer->buffer[ 1 ] = header % 256;

      buffer->total    = 2;
      buffer->refcount = 0;

      if ( ms_queue_each_client( items, conns, buffer, fd, off, flen ))
      {
         if ( ! buffer->refcount )
         {
            free( buffer->buffer );
            free( buffer );
         }

         return 1;
      }

      total -= MAX_FRAME;
      off   += MAX_FRAME;
   }

   return 0;
}

int ms_queue_sendfile_frame( struct ms_ccb **items, int conns, int final, unsigned int dlen, unsigned char *data, int fd, ssize_t offset, ssize_t slen )
{
   int i;
   unsigned int code, header;
   struct ms_buffer *buffer;
   struct ms_ccb **ptr;

   if ( ! conns )
      return 0;

   if (( dlen + slen ) > MAX_FRAME )
      return 1;

   for( i = 0, ptr = items; i < conns; ++i, ++ptr )
   {
      BIO *bio;

      if (( *ptr )->closed )
         continue;

      if (( bio = SSL_get_wbio(( *ptr )->ssl )) != NULL && ! BIO_get_ktls_send( bio ))
         return 2;
   }

   if (( buffer = ms_memory( sizeof( struct ms_buffer ))) == NULL )
      return 1;

   if (( buffer->buffer = ms_memory( dlen + 2 )) == NULL )
   {
      free( buffer );
      return 1;
   }

   code   = ( final ? TERMINATOR : 0 );
   header = code + dlen + slen;

   buffer->buffer[ 0 ] = header / 256;
   buffer->buffer[ 1 ] = header % 256;

   bcopy( data, &buffer->buffer[ 2 ], dlen );

   buffer->total    = dlen + 2;
   buffer->refcount = 0;

   if ( ms_queue_each_client( items, conns, buffer, ( slen ? fd : 0 ), offset, slen ))
   {
      if ( ! buffer->refcount )
      {
         free( buffer->buffer );
         free( buffer );
      }

      return 1;
   }

   return 0;
}

void ms_clear_events( struct ms_ccb *item )
{
   int i;

   for( i = ms_idx + 1; i < ms_out; ++i )
      if ( ms_outqueue[ i ].udata == item )
         ms_outqueue[ i ].ident = 0;
}

void ms_free_conn( struct ms_ccb *item )
{
   ms_clear_events( item );
   ms_close_callback( item );
   --ms_conn_count;

   if ( item->ssl != NULL )
   {
      SSL_shutdown( item->ssl );
      SSL_free( item->ssl );
   }

   close( item->sock );

   if ( item->buffer != NULL )
      free( item->buffer );

   while( item->first != NULL )
   {
      item->last = item->first->next;

      if ( ! --item->first->buffer->refcount )
      {
         free( item->first->buffer->buffer );
         free( item->first->buffer );
      }

      free( item->first );
      item->first = item->last;
   }

   free( item );
}

void ms_close_conn( struct ms_ccb *item, int now )
{
   if ( now || ! item->qlen )
      ms_free_conn( item );
   else
      ++item->closed;
}

void ms_ssl_accept( struct ms_ccb *item )
{
   int err;

   if (( err = SSL_accept( item->ssl )) <= 0 )
   {
      ms_set_events( item, err, -1 );
      return;
   }

   ms_stop_write_socket( item );
   ms_start_read_socket( item );

   item->stage = 0;

   if ( ms_open_callback( item ))
   {
      ms_free_conn( item );
      return;
   }
}

void ms_add_conn( int new )
{
   struct ms_ccb *ptr;

   if (( ptr = ms_memory( sizeof( struct ms_ccb ))) == NULL )
   {
      close( new );
      return;
   }

   bzero( ptr, sizeof( struct ms_ccb ));
   ms_close_on_exec( new );

   ptr->sock = new;
   ptr->stage = 255;

   if (( ptr->ssl = SSL_new( ctx )) == NULL )
   {
      ms_log_error();
      close( new );
      free( ptr );
      return;
   }

   if ( ! SSL_set_fd( ptr->ssl, ptr->sock ))
   {
      ms_log_error();
      SSL_free( ptr->ssl );
      close( new );
      free( ptr );
      return;
   }

   ++ms_conn_count;

   ms_ev_set( ptr->sock, EVFILT_READ, EV_ADD | EV_ENABLE, ptr );
   ms_ev_set( ptr->sock, EVFILT_WRITE, EV_ADD | EV_DISABLE, ptr );

   if ( ms_accept_callback( ptr ))
   {
      --ms_conn_count;
      SSL_free( ptr->ssl );
      close( new );
      free( ptr );
      return;
   }
}

void ms_accept_connection()
{
   int conn;

   while(( conn = accept( ms_fd, NULL, NULL )) > 0 )
   {
      if ( ms_conn_count == ms_max_conn )
      {
         close( conn );
         return;
      }

      ms_add_conn( conn );
   }

   if ( conn < 0 && errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
   {
      syslog( LOG_ERR, "accept(): %m" );
      exit( 1 );
   }
}

void ms_set_events( struct ms_ccb *item, int i, int reading )
{
   int err = SSL_get_error( item->ssl, i );

   ms_clear_events( item );

   if ( err == SSL_ERROR_WANT_READ )
   {
      item->opposite = ! reading;
      ms_stop_write_socket( item );
      ms_start_read_socket( item );
      return;
   }

   if ( err == SSL_ERROR_WANT_WRITE )
   {
      item->opposite = reading;
      ms_stop_read_socket( item );
      ms_start_write_socket( item );
      return;
   }

   if ( err == SSL_ERROR_SSL || err == SSL_ERROR_SYSCALL )
   {
      ms_log_error();
      SSL_free( item->ssl );
      item->ssl = NULL;
   }

   ms_free_conn( item );
}

void ms_transfer_out( struct ms_ccb *item )
{
   int count;
   struct ms_queue *q;

   if ( item->first == NULL )
      return;

   if ( ! item->first->finished && ( count = SSL_write( item->ssl, item->first->buffer->buffer, item->first->buffer->total )) <= 0 )
   {
      ms_set_events( item, count, 0 );
      return;
   }

   if ( item->first->sendfile )
   {
      ssize_t written;

      if ( ! item->first->finished )
         ++item->first->finished;

      if (( written = SSL_sendfile( item->ssl, item->first->sendfile, item->first->offset, item->first->len, 0 )) < 0 )
      {
         ms_set_events( item, written, 0 );
         return;
      }

      item->first->len    -= written;
      item->first->offset += written;
   }

   item->opposite = 0;

   ms_start_read_socket( item );
   ms_start_write_socket( item );

   if ( item->first->sendfile && item->first->len )
      return;

   q           = item->first;
   item->first = item->first->next;

   if ( ! --q->buffer->refcount )
   {
      free( q->buffer->buffer );
      free( q->buffer );
   }

   free( q );
   --item->qlen;

   if ( item->first == NULL )
   {
      item->last = NULL;

      ms_stop_write_socket( item );

      if ( item->closed )
         ms_free_conn( item );
      else
         ms_write_callback( item );
   }
}

void ms_transfer_in( struct ms_ccb *item )
{
   int i;
   unsigned char h;

AGAIN:
   if ( item->closed )
      return;

   switch( item->stage )
   {
      case 0:
         if (( i = SSL_read( item->ssl, &h, 1 )) <= 0 )
         {
            ms_set_events( item, i, 1 );
            return;
         }

         item->type = h & 0x80;
         item->len  = h & 0x7F;

         ++item->stage;

      case 1:
         if (( i = SSL_read( item->ssl, &h, 1 )) <= 0 )
         {
            ms_set_events( item, i, 1 );
            return;
         }

         if (( item->len = ( item->len << 8 ) + h ) > MAX_FRAME )
         {
            ms_free_conn( item );
            return;
         }

         if ( item->len )
         {
            if ( item->buffer == NULL )
            {
               if (( item->buffer = ms_memory( item->len )) == NULL )
               {
                  ms_free_conn( item );
                  return;
               }
            }
            else if (( item->offset + item->len ) > ms_max_message || ( item->buffer = realloc( item->buffer, item->offset + item->len )) == NULL )
            {
               ms_free_conn( item );
               return;
            }
         }

         ++item->stage;

      case 2:
         if ( item->len && ( i = SSL_read( item->ssl, &item->buffer[ item->offset ], item->len )) <= 0 )
         {
            ms_set_events( item, i, 1 );
            return;
         }

         item->opposite = 0;
         ms_start_read_socket( item );

         if ( item->qlen )
            ms_start_write_socket( item );
         else
            ms_stop_write_socket( item );

         if ( item->len )
         {
            item->len    -= i;
            item->offset += i;
         }

         if ( item->len )
            goto NEXT;

         item->stage = 0;

         if ( ! item->type )
            goto NEXT;

         if ( ms_read_callback( item, item->offset, item->buffer ))
            return;

         if ( item->buffer != NULL )
            free( item->buffer );

         item->buffer = NULL;
         item->offset = 0;

      NEXT:
         if ( SSL_has_pending( item->ssl ))
            goto AGAIN;
   }
}

void ms_sigterm_handler( int signo )
{
   ++ms_sigterm;
}

/*
 * If we have dropped a connection, we traverse the output kqueue and mark
 * any events for that connection as invalid by setting the socket
 * descriptor to 0.
 */

void ms_set_sigterm_intr()
{
   struct sigaction sigact;

   sigact.sa_handler = ms_sigterm_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGTERM, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction(): %m" );
}

void ms_process_clients()
{
   int kq;
   struct kevent *eptr;

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   ms_ev_set( ms_fd, EVFILT_READ, EV_ADD | EV_ENABLE, NULL );

   for( ; ; )
   {
      if ( ms_sigterm )
      {
         close( kq );
         break;
      }

      ms_set_sigterm_intr();
      ms_out = kevent( kq, ms_inqueue, ms_in, ms_outqueue, ms_qlen, NULL );
      ms_in = 0;

      if ( ms_out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %m" );
         exit( 1 );
      }

      signal( SIGTERM, ms_sigterm_handler );

      for( ms_idx = 0, eptr = &ms_outqueue[ ms_idx ]; ms_idx < ms_out; ++ms_idx, ++eptr )
      {
         struct ms_ccb *item;

         if ( ms_sigterm )
            break;

         if ( ! eptr->ident || eptr->flags & EV_ERROR )
            continue;

         item = ( struct ms_ccb *)eptr->udata;

         if ( eptr->filter == EVFILT_TIMER )
         {
            if ( ms_periodic != NULL )
               ms_periodic();

            continue;
         }

         if ( eptr->ident == ms_fd )
         {
            ms_accept_connection();
            continue;
         }

         if ( item->stage == 255 )
         {
            ms_ssl_accept( item );
            continue;
         }

         if ( eptr->filter == EVFILT_READ )
         {
            if ( item->opposite )
               ms_transfer_out( item );
            else
               ms_transfer_in( item );

            continue;
         }

         if ( item->opposite )
            ms_transfer_in( item );
         else
            ms_transfer_out( item );
      }
   }
}

int ms_password_callback( char *buf, int size, int rw, void *unused )
{
   char *ptr1, *ptr2;
   int len;

   for( len = 0, ptr1 = ms_password; *ptr1; ++ptr1 )
      ++len;

   if ( len + 1 > size )
   {
      *buf = '\0';
      fprintf( stderr, "%s: TLS password is too long for OpenSSL", ms_app_name );
      return 0;
   }

   ptr1 = ms_password;
   ptr2 = buf;

   while( *ptr1 )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';

   return len;
}

void ms_init_tls()
{
   static char context[ SSL_MAX_SSL_SESSION_ID_LENGTH ];
   int len;

   SSL_load_error_strings();
   SSL_library_init();

   if (( ctx = SSL_CTX_new( TLS_server_method() )) == NULL )
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_ecdh_auto( ctx, 1 ))
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_options( ctx, SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3 |
                                    SSL_OP_NO_TLSv1 | SSL_OP_NO_TLSv1_1 |
                                    SSL_OP_NO_TLSv1_2 |
                                    SSL_OP_NO_COMPRESSION |
                                    SSL_OP_ENABLE_KTLS |
                                    SSL_OP_SINGLE_DH_USE |
                                    SSL_OP_CIPHER_SERVER_PREFERENCE |
                                    SSL_OP_IGNORE_UNEXPECTED_EOF ))
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_clear_options( ctx, SSL_OP_LEGACY_SERVER_CONNECT |
                                      SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION ))
   {
      ms_log_error();
      exit( 1 );
   }

   SSL_CTX_set_default_passwd_cb( ctx, ms_password_callback );

   if ( ! SSL_CTX_use_PrivateKey_file( ctx, ms_keyfile, SSL_FILETYPE_PEM ))
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_use_certificate_chain_file( ctx, ms_chainfile ))
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_check_private_key( ctx ))
   {
      ms_log_error();
      exit( 1 );
   }

   len = snprintf( context, sizeof( context ), "message-%u", getpid() );

   if ( len >= sizeof( context ))
   {
      syslog( LOG_ERR, "libmessage generated an oversized TLS context. "
                       "This should not happen!" );
      exit( 1 );
   }

   if ( ! SSL_CTX_set_session_id_context( ctx, ( const unsigned char *)context, len ))
   {
      ms_log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_session_cache_mode( ctx, SSL_SESS_CACHE_SERVER ))
   {
      ms_log_error();
      exit( 1 );
   }

   SSL_CTX_set_verify( ctx, SSL_VERIFY_NONE, NULL );
}

char *ms_read_line( int fd )
{
   char c, *ptr;
   static char buffer[ 1024 ];
   int r, count;

   count = 0;
   ptr = buffer;

   for( ; ; )
   {
      if (( r = read( fd, &c, 1 )) <= 0 )
      {
         if ( r < 0 )
            fprintf( stderr, "%s: read(): %s\n", ms_app_name, strerror( errno ));
         else if ( count )
            return buffer;

         return NULL;
      }

      if ( c == 10 )
         break;

      if ( c == 13 )
         continue;

      ++count;
      *ptr++ = c;
      *ptr = '\0';

      if ( count == sizeof( buffer ) - 1 )
         break;
   }

   return buffer;
}

void ms_read_tls_config()
{
   int fd;
   char *ptr;

   if (( fd = open( ms_tls_file, O_RDONLY )) < 0 )
   {
      fprintf( stderr, "%s: Could not open: %s\n", ms_app_name, ms_tls_file );
      exit( 1 );
   }

   if (( ptr = ms_read_line( fd )) == NULL )
   {
      close( fd );
      fprintf( stderr, "%s: Could not read TLS keyfile from %s\n", ms_app_name, ms_tls_file );
      exit( 1 );
   }

   ms_keyfile = ms_str_dup( ptr );

   if (( ptr = ms_read_line( fd )) == NULL )
   {
      close( fd );
      fprintf( stderr, "%s: Could not read TLS key password from %s\n", ms_app_name, ms_tls_file );
      exit( 1 );
   }

   ms_password = ms_str_dup( ptr );

   if (( ptr = ms_read_line( fd )) == NULL )
   {
      close( fd );
      fprintf( stderr, "%s: Could not read TLS chainfile path from %s\n", ms_app_name, ms_tls_file );
      exit( 1 );
   }

   ms_chainfile = ms_str_dup( ptr );
   close( fd );
}

void ms_init( int argc, char **argv )
{
   ms_set_signals();
   ms_set_options( argc, argv );
   ms_read_tls_config();
   ms_init_tls();

   ms_qlen = ms_max_conn * MAX_EVENT;

   if (( ms_inqueue = ms_memory( sizeof( struct kevent ) * ms_qlen )) == NULL )
      exit( 1 );

   bzero( ms_inqueue, sizeof( struct kevent ) * ms_qlen );

   if (( ms_outqueue = ms_memory( sizeof( struct kevent ) * ms_qlen )) == NULL )
      exit( 1 );

   bzero( ms_outqueue, sizeof( struct kevent ) * ms_qlen );
}

int main( int argc, char **argv )
{
   ms_init( argc, argv );
   ms_init_func();

   if ( chdir( ms_rootdir ) < 0 )
   {
      fprintf( stderr, "chdir( %s ): %s\n", ms_rootdir, strerror( errno ));
      exit( 1 );
   }

   openlog( ms_app_name, LOG_PID, LOG_DAEMON );
   ms_logging = 1;

   if ( ! ms_testing )
      ms_become_daemon( argv );

   ms_start_listening();
   ms_change_identity();
   ms_process_clients();
   ms_exit_func();

   return 0;
}
