; Copyright (c) 2004, 2010, James Bailie <jimmy@mammothcheese.ca>.
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions are met:
;
;     * Redistributions of source code must retain the above copyright
; notice, this list of conditions and the following disclaimer.
;     * Redistributions in binary form must reproduce the above copyright
; notice, this list of conditions and the following disclaimer in the
; documentation and/or other materials provided with the distribution.
;     * The name of James Bailie may not be used to endorse or promote
; products derived from this software without specific prior written permission.
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
; AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
; ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
; CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.

; This module extracts single-character command-line options and maps the
; options to their values in a table named "options", leaving the argument
; pointer pointing at the first argument after the options.  No type
; conversion is performed on values extracted from the command line.  They
; are stored in the "options" table as strings.  Boolean options, however,
; which have no values, will be mapped to the number 1, if found.  Option
; characters must be in the set [A-Za-z0-9].

; A single function named "getopt" intializes the "options" table.  The lone
; argument to the function is a stack containing option strings that the
; user wishes to recognize as boolean options which are either present or
; not present.  Boolean options will be mapped to 1 in the "options" table
; if they are found, otherwise there will be no mapping for the boolean
; option.  This holds true for all options.  The absence of a mapping
; indicates the option's absence from the command line.  If no booleans are
; to be recognized 0 may be passed as the argument to "getopt".

; Options are recognized in either -ovalue or -o value formats.  To pass an
; initial command-line argument which looks like an option to a program using
; this module, the option list must be terminated with the end of option flag
; "--".  If the argument is a filename, one could also fully-qualify it.  For
; example:

; % booger.mm -n10 -f 11 -- -weird-filename

; or

; % booger.mm -n10 -f 11 ./-weird-filename

(setq getopt_find
   (lambda (str stk)
      (getopt_find_helper str 0 (used stk) stk)))

(setq getopt_find_helper
   (lambda (str idx last stk)
      (when (< idx last)
         (if (eq str (index stk idx))
            1
            (getopt_find_helper str (+ idx 1) last stk)))))

(setq options (table))
(setq getopt_rx (regcomp "^-([A-Za-z0-9])?(.*)"))

(setq getopt
   (lambda (boolean)
      (call_cc
         (lambda (k)
            (getopt_helper k (next) boolean)))))

(setq getopt_helper
   (lambda (k arg boolean)
      (when arg

         (letn ((m (matches getopt_rx arg))
                (item (and (stackp m) (index m 1)))
                (val (and (stackp m) (index m 2))))

            (if (not m)
               (previous)

               (if (not item)
                  (if (eq val "-")
                     (k 0)
                     (warn "malformed option: " (current)))

                  (if val
                     (hash options item val)

                     (if (and (stackp boolean) (getopt_find item boolean))
                        (hash options item 1)
                        (hash options item (next)))))

               (getopt_helper k (next) boolean))))))
