/*
 * Copyright (c) 2004-2011, James Bailie.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote 
 * products derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <libgen.h>
#include <dirent.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <regex.h>
#include <sqlite3.h>

#include "functions.h"

#define HASH_SIZE 512
#define POOL_INC 2048
#define GC_FREQUENCY 500

sqlite3 *db = NULL;

int gc = GC_FREQUENCY;

#define type(n) ( n & 127 )
#define mark(n) ( n |= 128 )
#define unmark(n) ( n &= -129 )
#define ismarked(n) ( n & 128 )

#define length(n) ((int)n >> 8)
#define bzero(n,m) { char *ptr; int i; for( ptr = ( char *)n, i = m; i; --i ) *ptr++ = '\0'; }

#define GLOBAL(n) globals[ n ]
#define LOCAL(n) stack->values[ n ]
#define CLOSURE_REF(n) ( *stack->top )->data.closure[ n ]
#define TOS *stack->top
#define PUSH(n) { if ( stack->free ) { if ( stack->used ) ++stack->top; --stack->free; ++stack->used; *stack->top = ( n ); } else stack_push( stack, n ); }
#define MAKE_STRING(s,l) make_atom_from_string( s, l, 1 )
#define MAKE_NUMBER(n) make_atom_from_number(n)
#define MAKE_CLOSURE(f,n) make_closure( &&FUNCTION_ ## f , n )
#define HALT return
#define EXIT exit_status = ( stack_pop( stack ))->data.number; return
#define RESET_STACK stack->free += stack->used; stack->top = stack->values; stack->used = 0
#define JUMP if ( ! --gc ) { collect_garbage(); gc = GC_FREQUENCY; } goto *stack->values[ 0 ]->data.closure[ 0 ];
#define BEGIN_IF { struct atom *atom = stack_pop( stack ); \
   if (( type( atom->flags ) == ATOM_NUMBER ? atom->data.number : ( atom != empty_string_atom ))) {
#define ELSE } else {
#define END_IF } }

#define ATOM_STRING  1
#define ATOM_CLOSURE 2
#define ATOM_STACK   3
#define ATOM_REGEXP  4
#define ATOM_TABLE   5
#define ATOM_RECORD  6
#define ATOM_NUMBER  7
#define ATOM_SQL     8

struct string
{
   int free, used;
   char *top;
   char *str;
};

struct hash_elt
{
   struct atom *element, *binding;
   struct hash_elt *next;
};

struct atom
{
   char *syntax;
   int flags;

   union {
      struct atom **closure;
      struct stack *stack;
      struct hash_elt **table;
      regex_t *regexp;
      struct atom **record;
      int number;
      sqlite3_stmt *sql;
   } data;
};

struct stack
{
   int free, used;
   struct atom **top, **values;
};

struct int_stack
{
   int free, used;
   int *top, *values;
};

struct atom *empty_string_atom, *atom_pool, *atom_pool_ptr, **globals;
struct string *working_string, *private_string;
struct stack *stack, *reclaimed_atoms, *atom_pool_stack, *working_stack;

struct int_stack *descriptors[ 3 ];

char **first_arg, **last_arg, **arg_ptr;

struct hash_elt *atoms[ HASH_SIZE ];

int atom_pool_free, stack_inc, exit_status;

void *memory( int );

struct hash_elt **make_table();
struct atom *lookup_elt( struct hash_elt **, struct atom * );
void insert_elt( struct hash_elt **, struct atom *, struct atom * );
void remove_elt( struct hash_elt **, struct atom * );
struct stack *get_hash_keys( struct hash_elt ** );
struct stack *get_hash_values( struct hash_elt ** );

struct stack *make_stack();
struct atom *stack_pop( struct stack * );
void stack_push( struct stack *, struct atom * );
void stack_truncate( struct stack *, int );
void stack_clear( struct stack * );

int stack_pop_int( struct int_stack * );
void stack_push_int( struct int_stack *, int );

struct string *make_string();
void string_append( struct string *, char );
void string_prepend( struct string *, char );
void string_erase( struct string *, int );
void string_clear( struct string * );
void string_assign( struct string *, char *, int );
void string_chop( struct string * );

struct atom *make_atom();
struct atom *make_atom_from_string( char *, int, int );
struct atom *make_atom_from_number( int );
struct atom *make_atom_from_stack( struct stack * );
struct atom *make_atom_from_record( struct atom ** );
struct atom *make_atom_from_regexp( regex_t * );
struct atom *make_atom_from_table( struct hash_elt ** );

void mark_stack( struct atom * );
void mark_closure( struct atom * );
void mark_atom( struct atom * );
void mark_table( struct atom * );

void make_closure( void *, int );

char *str_dup( char *, int );

void resume_descriptor( int );

/*
 * Intrinsics-related.
 */

#define REDIRECT redirect()
#define RESUME resume_descriptor(( *stack->top )->data.number )
#define PIPE { int arg1; struct atom *arg2; arg2 = stack_pop( stack ); arg1 = ( *stack->top )->data.number; \
   pipe_open( "pipe", arg2->syntax, arg1 ); }

#define MIN(a,b) ( a < b ? a : b )

#define CURRENT stack_push( stack, make_atom_from_string( *arg_ptr, -1, 1 ))
#define NEXT stack_push( stack, ( arg_ptr < last_arg ? make_atom_from_string( *++arg_ptr, -1, 1 ) : make_atom_from_number( 0 )))
#define PREVIOUS stack_push( stack, ( arg_ptr > first_arg ? make_atom_from_string( *--arg_ptr, -1, 1 ) : make_atom_from_number( 0 )))
#define REWIND stack_push( stack, make_atom_from_string( *( arg_ptr = first_arg ), -1, 1 ))

#define EQ { struct atom *op2 = stack_pop( stack ); *stack->top = make_atom_from_number( *stack->top == op2 ); }

#define LT { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number < op2->data.number ); }

#define GT { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number > op2->data.number ); }

#define GTE { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number >= op2->data.number ); }

#define LTE { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number <= op2->data.number ); }

#define ADD { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number + op2->data.number ); }

#define SUB { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number - op2->data.number ); }

#define DIV { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number / op2->data.number ); }

#define MUL { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number * op2->data.number ); }

#define MOD { struct atom *op2 = stack_pop( stack ); \
   *stack->top = make_atom_from_number(( *stack->top )->data.number % op2->data.number ); }

#define ABS *stack->top = make_atom_from_number( abs(( *stack->top )->data.number ))

#define FORK stack_push( stack, make_atom_from_number( fork() ));

#define STACK { int i = ( *stack->top )->data.number; *stack->top = make_atom_from_stack( make_stack() ); \
   while ( i-- ) stack_push( ( *stack->top )->data.stack, empty_string_atom ); }
#define USED *stack->top = make_atom_from_number(( *stack->top )->data.stack->used )

#define INDEX { int i = ( stack_pop( stack ))->data.number; struct stack *s = ( *stack->top )->data.stack; \
   *stack->top = s->values[ i ]; }

#define TOPIDX { struct stack *s = ( *stack->top )->data.stack; *stack->top = make_atom_from_number( s->used - 1 ); }

#define STORE { struct atom *atom = stack_pop( stack ); int i = ( stack_pop( stack ) )->data.number; \
   struct stack *s = ( *stack->top )->data.stack; s->values[ i ] = atom; }

#define STACK_PUSH { struct atom *atom = stack_pop( stack ); struct stack *s = ( *stack->top )->data.stack; \
   stack_push( s, atom ); }

#define STACK_POP *stack->top = stack_pop( ( *stack->top )->data.stack );

#define CLEAR ( *stack->top )->data.stack->free += ( *stack->top )->data.stack->used; \
   ( *stack->top )->data.stack->top = ( *stack->top )->data.stack->values; \
   ( *stack->top )->data.stack->used = 0;

#define SORT_NUMBERS qsort( ( *stack->top )->data.stack->values, ( *stack->top )->data.stack->used, sizeof( struct atom * ), compare_numbers )
#define SORT_STRINGS qsort( ( *stack->top )->data.stack->values, ( *stack->top )->data.stack->used, sizeof( struct atom * ), compare_strings )

#define LENGTH *stack->top = make_atom_from_number( length( ( *stack->top )->flags ));

#define CHAR { char c = ( char )( *stack->top )->data.number; *stack->top = make_atom_from_string( &c, 1, 1 ); }

#define CODE { char c = *( *stack->top )->syntax; *stack->top = make_atom_from_number( c ); }

#define STRINGIFY { char n[ 32 ]; snprintf( n, sizeof( n ), "%d", ( *stack->top )->data.number ); \
   *stack->top = make_atom_from_string( n, -1, 1 ); }

#define DIGITIZE { int n = strtol( ( *stack->top )->syntax, NULL, 10 ); *stack->top = make_atom_from_number( n ); }

#define CHOP string_assign( working_string, ( *stack->top )->syntax, length(( *stack->top )->flags )); \
   string_chop( working_string ); *stack->top = make_atom_from_string( working_string->str, working_string->used, 1 );

#define CHOMP { char *ptr; string_assign( working_string, ( *stack->top )->syntax, length(( *stack->top )->flags )); \
   for( ptr = ( working_string->top - 1 ); ptr >= working_string->str; --ptr ) \
      if ( *ptr == '\n' || *ptr == '\r' ) string_chop( working_string ); \
   *stack->top = make_atom_from_string( working_string->str, working_string->used, 1 ); }

#define EXPLODE { struct stack *output = make_stack(); char *ptr; \
   string_assign( working_string, ( *stack->top )->syntax, length( ( *stack->top )->flags )); \
   for( ptr = working_string->str; ptr < working_string->top; ++ptr ) \
      stack_push( output, make_atom_from_string( ptr, 1, 1 )); \
   *stack->top = make_atom_from_stack( output ); }

#define SUBSTRING { int len1, len2, start; len2 = ( stack_pop( stack ))->data.number; \
   start = ( stack_pop( stack ))->data.number; \
   len1 = length( ( *stack->top )->flags ) - start; \
   *stack->top = make_atom_from_string( &( *stack->top )->syntax[ start ], \
      ( len2 ? MIN( len1, len2 ) : len1 ), 1 ); }

#define CONCAT { int len, len2, len3; char *c; struct atom **ptr; \
   len = ( stack_pop( stack ))->data.number; ptr = ( stack->top - ( len3 = --len )); \
   string_assign( working_string, ( *ptr )->syntax, length( ( *ptr )->flags ) ); \
   while( len-- ) { c = ( *++ptr )->syntax; len2 = length( ( *ptr )->flags ); \
   while( len2-- ) string_append( working_string, *c++ ); } \
   while( len3-- ) stack_pop( stack ); \
   *stack->top = make_atom_from_string( working_string->str, working_string->used, 1 ); }

#define GETLINE { char buffer[ 4096 ]; *buffer = '\0'; \
   if ( fgets( buffer, sizeof( buffer ), stdin ) == NULL ) { \
      if ( feof( stdin ) ) stack_push( stack, make_atom_from_number( 0 )); \
      else { fprintf( stderr, "getline: %s\n", strerror( errno )); \
         stack_push( stack, empty_string_atom ); } } \
   else stack_push( stack, make_atom_from_string( buffer, strlen( buffer ), 1 ) ); }

#define NUMBERP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_NUMBER )
#define STRINGP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_STRING )
#define STACKP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_STACK )
#define TABLEP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_TABLE )
#define REGEXPP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_REGEXP )
#define SQLP *stack->top = make_atom_from_number( type( ( *stack->top )->flags ) == ATOM_SQL )

#define SETENV { char *val = ( stack_pop( stack ) )->syntax; char *name = ( *stack->top )->syntax; \
   *stack->top = make_atom_from_number( setenv( name, val, 1 ) ); }

#define GETENV { char *env = getenv( ( *stack->top )->syntax ); \
   *stack->top = ( env == NULL ? make_atom_from_number( 0 ) : make_atom_from_string( env, -1, 1 ) ); }

#define SYSTEM *stack->top = make_atom_from_number( system( ( *stack->top )->syntax ) )

#define TABLE stack_push( stack, make_atom_from_table( make_table() ))
#define HASH { struct atom *obj = stack_pop( stack ), *key = stack_pop( stack ); \
   insert_elt( ( *stack->top )->data.table, key, obj ); }
#define UNHASH { struct atom *key = stack_pop( stack ); remove_elt( ( *stack->top )->data.table, key ); }
#define LOOKUP { struct atom *key = stack_pop( stack ); *stack->top = lookup_elt( ( *stack->top )->data.table, key ); }
#define KEYS *stack->top = make_atom_from_stack( get_hash_keys( ( *stack->top )->data.table ))
#define VALUES *stack->top = make_atom_from_stack( get_hash_values( ( *stack->top )->data.table ))

#define SHIFT shift()
#define UNSHIFT unshift()
#define JOIN join()
#define SPLIT split()
#define EXEC exec()
#define SUBSTACK substack()
#define APPEND append_stacks()

#define STRCMP { struct atom *a = stack_pop( stack ); \
   *stack->top = make_atom_from_number( strcmp( ( *stack->top )->syntax, a->syntax ) ); }

#define REG_COMP regexp_comp()
#define REG_MATCH regexp_match( 0 )
#define REG_MATCHES regexp_match( 1 )
#define REG_SUBST regexp_substitute()

#define BASENAME *stack->top = make_atom_from_string( basename( ( *stack->top )->syntax ), -1, 1 )
#define DIRNAME *stack->top = make_atom_from_string( dirname( ( *stack->top )->syntax ), -1, 1 )

#define RANDOM *stack->top = make_atom_from_number( ( int )( random() * ( double )( *stack->top )->data.number / RAND_MAX ))

#define DIRECTORY directory()
#define RENAME file_rename()
#define REMOVE file_remove()
#define RMDIR dir_remove()
#define STAT file_stat()
#define SYMLINK file_symlink()

#define READCHARS readchars()
#define EXPAND_TABS expand_tabs()

#define MAKE_RECORD { int n = ( *stack->top )->data.number; \
   struct atom **ptr = ( struct atom **)memory( sizeof( struct atom * ) * n ); \
   *stack->top = make_atom_from_record( ptr ); }
#define GETFIELD { int n = ( stack_pop( stack ))->data.number; *stack->top = ( *stack->top )->data.record[ n ]; }
#define SETFIELD { struct atom *obj = stack_pop( stack ); int n = ( stack_pop( stack ))->data.number; \
   *stack->top = ( *stack->top )->data.record[ n ] = obj; }

#define PRINT { int n = ( stack_pop( stack ))->data.number, m = n; \
   while( --n >= 0 ) fputs( ( *( stack->top - n ))->syntax, stdout ); \
   stack->free += m; stack->used -= m; stack->top -= m; \
   if ( stack->top < stack->values ) stack->top = stack->values; \
   stack_push( stack, make_atom_from_number( 1 )); }

#define PRINTLN { int n = ( stack_pop( stack ))->data.number, m = n; \
   while( --n >= 0 ) fputs( ( *( stack->top - n ))->syntax, stdout ); \
   stack->free += m; stack->used -= m; stack->top -= m; \
   if ( stack->top < stack->values ) stack->top = stack->values; \
   fputc( '\n', stdout ); \
   stack_push( stack, make_atom_from_number( 1 )); }

#define FLUSH stack_push( stack, make_atom_from_number( fflush( stdout )))

#define WARN { int n = ( stack_pop( stack ))->data.number, m = n; \
   while( --n >= 0 ) fputs( ( *( stack->top - n ))->syntax, stderr ); \
   stack->free += m; stack->used -= m; stack->top -= m; \
   if ( stack->top < stack->values ) stack->top = stack->values; \
   fputc( '\n', stderr ); \
   stack_push( stack, make_atom_from_number( 1 )); }

#define DIE { int n = ( stack_pop( stack ))->data.number, m = n; \
   while( --n >= 0 ) fputs( ( *( stack->top - n ))->syntax, stderr ); \
   stack->free += m; stack->used -= m; stack->top -= m; \
   if ( stack->top < stack->values ) stack->top = stack->values; \
   fputc( '\n', stderr ); exit( 1 ); }

#define TIME { time_t t = time( NULL ); char buffer[ 32 ]; \
   snprintf( buffer, sizeof( buffer ), "%016lu", t ); \
   stack_push( stack, make_atom_from_string( buffer, -1, 1 )); }

#define DATE { time_ t = time( NULL ); struct tm *lt = gmtime( &t ); char buffer[ 64 ]; \
   strftime( buffer, sizeof( buffer ), "%a, %d %b %Y %H:%M:%S %Z", lt ); \
   stack_push( stack, make_atom_fron_string( buffer, -1, 1 )); }

#define OPEN sqlite_open( ( *stack->top )->syntax )
#define CLOSE sqlite_close( db )
#define SEXEC sqlite_exec( ( *stack->top )->syntax )

int compare_numbers( const void *, const void * );
int compare_strings( const void *, const void * );

void join();
void split();

void exec();
void regexp_comp();
void regexp_match( int );
void regexp_substitute();

void redirect();
void pipe_open( char *, char *, int );
void shift();

void readchars();
void directory();
void file_symlink();
void file_stat();
void file_remove();
void file_rename();
void expand_tabs();

void sqlite_open( char * );
void sqlite_close();
void sqlite_exec( char * );

void sqlite_prepare();
void sqlite_bind();
void sqlite_step();
void sqlite_row();
void sqlite_reset();
void sqlite_finalize();
