/*
 * Nellie Copyright (c) 2017-2023, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "formatter.h"

#include <dirent.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>

extern void *memory( int );
extern char *str_dup( char *, int );

extern int SCENE_ORPH, CUE_ORPH, LINES, CHARS, FULL_LEN, QUARTER, HALF, TOP, WIDTH, LENGTH;

int epub_normalize_whitespace( unsigned char *input, struct string *source )
{
   int len, ignore, last, space, underscore = 0, hyphen = 0;
   unsigned char *ptr;

   for( ptr = input; *ptr; ++ptr )
      if ( ! isspace( *ptr ))
         break;

   for( ignore = 0, space = 1, last = 0; *ptr; ++ptr )
   {
      if ( *ptr == '~' )
         continue;

      if ( ! isspace( *ptr ))
      {
         last = 0;

         if ( space )
            ++space;

         if ( *ptr == '\\' && ( len = process_accent( ptr + 1, source )) > 0 )
         {
            ptr += len;
            continue;
         }
      }
      else if ( last )
         continue;
      else
      {
         ++last;
         ++space;
         *ptr = ' ';
      }

      switch( *ptr )
      {
         case '_':
            if ( !underscore && space == 2 )
               ++ignore;

            underscore ^= 1;
            if ( string_concat( source, ( underscore ? "<em>" : "</em>" )))
               return 1;
            break;

         case '-':
            if ( ++hyphen == 2 && source->used )
            {
               --source->top;
               ++source->free;
               --source->used;
               *source->top = '\0';

               if ( string_concat( source, "&#8212;" ))
                  return 1;

               hyphen = 0;
            }
            else if ( string_append( source, *ptr ))
               return 1;
            break;

         case '&':
            if ( string_concat( source, "&amp;" ))
               return 1;
            break;

         case '"':
            if ( string_concat( source, ( space || ignore ? "&#8220;" : "&#8221;" )))
               return 1;
            break;

         case '\'':
            if ( string_concat( source, ( space || ignore ? "&#8216;" : "&#8217;" )))
               return 1;
            break;

         case '`':
            if ( string_concat( source, "&#8217;" ))
               return 1;
            break;

         case '<':
            if ( string_concat( source, "&lt;" ))
               return 1;
            break;

         case '>':
            if ( string_concat( source, "&gt;" ))
               return 1;
            break;

         case '|':
            if ( string_concat( source, "<br/>" ))
               return 1;
            break;

         default:
            if ( string_append( source, *ptr ))
               return 1;
      }

      if ( *ptr != '-' )
         hyphen = 0;

      if ( space > 1 )
         space = 0;

      if ( ignore == 1 )
         ++ignore;
      else if ( ignore > 1 )
         ignore = 0;
   }

   return 0;
}

char *meta_escape( char *input )
{
   char *ptr;
   struct string *output;

   if (( output = string_make( NULL )) == NULL )
      exit( 1 );

   if ( input == NULL )
   {
      if ( string_concat( output, "UNKNOWN" ))
         exit( 1 );

      goto OK;
   }

   for( ptr = input; *ptr; ++ptr )
      if ( *ptr == ';' )
         *ptr = ' ';

   if ( epub_normalize_whitespace(( unsigned char * )input, output ))
      exit( 1 );

OK:
   ptr = output->str;
   free( output );
   return ptr;
}

struct stack *epub_wrap_lines( char *input )
{
   struct stack *para;
   struct string *line = NULL, *source = NULL;
   union stack_val val;
   unsigned char *ptr;
   int i, n, first = 1;

   for( n = 0, ptr = ( unsigned char *)input; *ptr; ++ptr )
      if ( *ptr == '_' )
         ++n;

   if ( n % 2 )
   {
      fprintf( stderr, "unbalanced underscores in: %s\n", input );
      exit( 1 );
   }

   if ( input == NULL || ( para = stack_make()) == NULL )
      return NULL;

   if (( source = string_make( NULL )) == NULL )
      goto ERR;

   if ( epub_normalize_whitespace(( unsigned char *)input, source ))
      goto ERR;

   if (( line = string_make( NULL )) == NULL )
      goto ERR;

   for( n = i = 0; source->str[ i ]; )
   {
      /*
       * Remove leading whitespace.
       */

      for( ptr = ( unsigned char *)&source->str[ i ]; *ptr; ++ptr, ++i )
         if ( *ptr != ' ' )
            break;

      for( n = ( first ? n : 0 ), ptr = ( unsigned char *)&source->str[ i ]; n < 80 && *ptr; ++ptr, ++i )
      {
         /*
          * Do not count underscores.  They will be removed.
          */

         if (( *ptr & 0xC0 ) != 0x80 && *ptr != '_' )
            ++n;

         if ( string_append( line, *ptr ))
            goto ERR;
      }

      /*
       * If we have not broken the paragraph at whitespace naturally,
       * move the break back until we do break at whitespace.
       */

      while( line->used && *ptr && *( ptr - 1 ) != ' ' && *ptr != ' ' )
      {
         --line->used;
         ++line->free;
         --line->top;

         line->str[ line->used ] = '\0';
         --ptr;
         --i;
      }

      /*
       * Remove trailing whitespace from the new line.
       */

      while( line->used && *--ptr == ' ' )
      {
         --line->used;
         ++line->free;
         --line->top;

         line->str[ line->used ] = '\0';
         --i;
      }

      /*
       * If we did not find whitespace to break at, we have a line with a
       * token in it that is longer than the line length.  We create an
       * overlong line by moving the break forward until we find whitespace
       * or we run out of data.
       */

      if ( ! line->used )
         for( ptr = ( unsigned char *)&source->str[ i ]; *ptr && *ptr != ' '; ++ptr, ++i )
            if ( string_append( line, *ptr ))
               goto ERR;

      /*
       * Do not create blank lines.  This can happen if input is all
       * whitespace.
       */

      if ( line->used )
      {
         if (( val.ptr = str_dup( line->str, line->used )) == NULL )
            goto ERR;

         if ( stack_push( para, val ))
         {
            free( val.ptr );
            goto ERR;
         }
      }

      string_clear( line );
      first = 0;
   }

   string_free( source );
   string_free( line );
   return para;

ERR:
   if ( source != NULL )
      string_free( source );

   if ( line != NULL )
      string_free( line );

   while( para->used )
   {
      union stack_val *vptr = stack_pop( para );
      free( vptr->ptr );
   }

   stack_free( para );
   return NULL;
}

struct stack *make_non_breaking()
{
   struct stack *para;
   union stack_val val;

   if (( para = stack_make()) == NULL )
      return NULL;

   if (( val.ptr = str_dup( "&#160;", 6 )) == NULL )
   {
      stack_free( para );
      return NULL;
   }

   if ( stack_push( para, val ))
   {
      free( val.ptr );
      stack_free( para );
      return NULL;
   }

   return para;
}

struct para *epub_make_para( char *data, enum paras type, char *last_cue, int scene_no, char *shot, int numbers, int addcont )
{
   struct para *pptr;
   int cont = 0;

   if (( pptr = memory( sizeof( struct para ))) == NULL )
      return NULL;

   pptr->type = type;
   pptr->shot = *shot;
   pptr->next = NULL;

   if ( addcont && ( type == CUE || type == LCUE || type == RCUE ) && ( same_cue( data, last_cue )))
   {
      ++cont;

      if (( data = add_cont( data )) == NULL )
      {
         free( pptr );
         return NULL;
      }
   }

   if ( type == IDIALOG && data[ 0 ] == ' ' && data[ 1 ] == '\0' )
      pptr->lines = make_non_breaking();
   else
      pptr->lines = epub_wrap_lines( data );

   if ( pptr->lines == NULL )
   {
      if ( cont )
         free( data );

      free( pptr );
      return NULL;
   }

   if (( type == SCENE || type == SHOT ) && numbers && add_scene_numbers( type, pptr->lines, scene_no, shot, 1 ))
   {
      union stack_val *vptr;

      while( pptr->lines->used )
      {
         vptr = stack_pop( pptr->lines );
         free( vptr->ptr );
      }

      stack_free( pptr->lines );
      free( pptr );
      pptr = NULL;
   }

   if ( cont )
      free( data );

   return pptr;
}

struct para *epub_make_para_list( struct tsml *tree, struct meta *meta, char *scenes, int numbers, int addcont )
{
   struct tsml *tptr;
   struct para *list = NULL, *pptr = NULL, **prev = NULL;
   enum paras type;
   int scene_no, last_scene, on = 0;
   char *last_cue = NULL, ellipsis[ 6 ] = ". . .";
   static char shot[] = "@";
   struct stack *scene_list = NULL;
   union stack_val val;

   if ( tree == NULL || tree->type != ROOT )
      return NULL;

   if ( scenes == NULL || ! *scenes )
      ++on;
   else if (( scene_list = make_scene_number_list( scenes )) == NULL )
      return NULL;

   /*
    * If the y element is present, make it root.
    */

   if ( tree->children            != NULL && tree->children->type == OPEN &&
        tree->children->data      != NULL &&
        tree->children->data[ 0 ] == 'y'  && ! tree->children->data[ 1 ] )
      tree = tree->children;

   for( last_scene = scene_no = 0, tptr = tree->children; tptr != NULL; tptr = tptr->next )
   {
      /*
       * Skip whitespace outside of elements.
       */

      if ( tptr->type == DATA )
         continue;

      if ( tptr->data == NULL )
         goto ERR;

      /*
       * Bail if the name of this element is more than 1 character in
       * length or if the element has no children.
       */

      if (( *tptr->data && *( tptr->data + 1 )) || tptr->children == NULL )
         break;

      switch( *tptr->data )
      {
         case 'v':
            type = ACT; last_cue = NULL;
            break;

         case 'e':
            type = END; last_cue = NULL;
            break;

         case 't':
            type = TRANS; last_cue = NULL;
            break;

         case 's':
            type = SCENE; last_cue = NULL;

            ++scene_no;
            *shot = '@';

            if ( scene_list != NULL && ( on = ( scene_list->used && scene_list->values[ 0 ].num == scene_no )))
            {
               if ( scene_no != last_scene + 1 )
               {
                  if (( pptr = epub_make_para( ellipsis, END, last_cue, 0, shot, 0, 0 )) == NULL )
                     goto ERR;

                  if ( list == NULL )
                     list = pptr;

                  if ( prev != NULL )
                     *prev = pptr;

                  prev = &pptr->next;
               }

               stack_shift( scene_list, &val );
               last_scene = scene_no;
            }
            break;

         case 'o':
            type = SHOT;

            if ( *shot < 'Z' )
               ++*shot;
            else if ( *shot == 'Z' )
               *shot = 'a';
            else if ( *shot < 'z' )
               ++*shot;
            break;

         case 'a':
            type = ACTION;
            break;

         case 'c':
            type = CUE;
            break;

         case 'f':
            type = LCUE;
            break;

         case 'r':
            type = RCUE;
            break;

         case 'd':
            type = DIALOG;
            break;

         case 'i':
            type = IDIALOG;
            break;

         case 'p':
            type = PAREN;
            break;

         case 'l':
            type = LYRIC;
            break;

         case 'm':
            if ( meta != NULL )
               extract_metadata( tptr, meta );
            continue;

         default:
            goto ERR;
      }

      if ( ! on )
         continue;

      /*
       * Bail if the element has more than one child or if the lone child
       * is not a data node.
       */

      if ( tptr->children->next != NULL || tptr->children->type != DATA )
            break;

      if (( pptr = epub_make_para( tptr->children->data, type, last_cue, scene_no, shot, numbers, addcont )) == NULL )
         break;

      if ( type == CUE || type == LCUE || type == RCUE )
         last_cue = tptr->children->data;

      if ( list == NULL )
         list = pptr;

      if ( prev != NULL )
         *prev = pptr;

      prev = &pptr->next;
   }

ERR:
   if ( scene_list != NULL )
      stack_free( scene_list );

   if ( tptr != NULL )
   {
      free_para_list( list );
      list = NULL;
   }

   return list;
}

void make_dir( char *path )
{
   int mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;

   if ( mkdir( path, mode ) < 0 )
   {
      fprintf( stderr, "mkdir( %s ): %s\n", path, strerror( errno ));
      exit( 1 );
   }
}

FILE *open_file( char *path )
{
   FILE *file;

   if (( file = fopen( path, "w" )) == NULL )
   {
      fprintf( stderr, "fopen( %s ): %s\n", path, strerror( errno ));
      exit( 1 );
   }

   return file;
}

void put_string( char *str, FILE *file )
{
   if ( fputs( str, file ) < 0 )
   {
      fprintf( stderr, "fputs( %s ): %s", str, strerror( errno ));
      exit( 1 );
   }
}

void concat( struct string *str, char *data )
{
   if ( string_concat( str, data ))
      exit( 1 );
}

void remove_dir( char *name )
{
   DIR *dir;
   struct dirent *entry;
   char buffer[ MAXPATHLEN + 1 ];

   if (( dir = opendir( name )) == NULL )
   {
      if ( errno == ENOENT )
         return;

      fprintf( stderr, "opendir( %s ): %s\n", name, strerror( errno ));
      exit( 1 );
   }

   readdir( dir );
   readdir( dir );

   while(( entry = readdir( dir )) != NULL )
   {
      snprintf( buffer, sizeof( buffer ), "%s/%s", name, entry->d_name );

      if ( unlink( buffer ) < 0 )
      {
         fprintf( stderr, "unlink( %s ): %s\n", buffer, strerror( errno ));
         exit( 1 );
      }
   }

   closedir( dir );

   if ( rmdir( name ) < 0 )
   {
      fprintf( stderr, "rmdir( %s ): %s", name, strerror( errno ));
      exit( 1 );
   }
}

void remove_ebook( char *root )
{
   char **ptr, buffer[ MAXPATHLEN + 1 ], *dirs[] = { "ebook/META-INF", "ebook/OEBPS", "ebook", NULL };

   for( ptr = dirs; *ptr != NULL; ++ptr )
   {
      snprintf( buffer, sizeof( buffer ), "%s/%s", root, *ptr );
      remove_dir( buffer );
   }
}

void make_dirs( char *root )
{
   char buffer[ MAXPATHLEN + 1 ],
        *mime_type = "application/epub+zip",

        *content_xml = "<?xml version=\"1.0\"?>\n"
                       "<container version=\"1.0\" xmlns=\"urn:oasis:names:tc:opendocument:xmlns:container\">\n"
                       "<rootfiles>\n"
                       "<rootfile full-path=\"OEBPS/content.opf\" media-type=\"application/oebps-package+xml\"/>\n"
                       "</rootfiles>\n"
                       "</container>",

    *styles = "h1 { font-size: 2em; font-weight: bold; text-align: center; }\n"
    "h2 { font-size: 1.5em; font-weight: bold; text-align: center; margin-top: 1em; }\n"
    "p { margin: 0; padding: 0; text-align: left; text-indent: 0; }\n"
    "p.copy { text-indent: 0; font-size: 0.8em; text-align: center; margin-top: 4em; }\n"
    "p.scene { font-size: 1em; font-weight: bold; margin-top: 0.25em; margin-bottom: 0.25em; }\n"
    "p.action { margin-top: 1em; }\n"

    "p.cue { margin-left: 5em; margin-top: 1em; page-break-after: avoid; }\n"
    "p.lcue { margin-right: 4em; margin-left: 3em; margin-top: 1em; page-break-after: avoid; }\n"
    "p.rcue { margin-left: 7em; margin-right: 0em; margin-top: 1em; page-break-after: avoid; }\n"

    "p.dialog  { margin-left: 2em; margin-right: 2em; }\n"
    "p.ldialog  { margin-right: 4em; margin-left: 0em; }\n"
    "p.rdialog  { margin-left: 4em; margin-right: 0em; }\n"

    "p.idialog { margin-left: 2em; margin-right: 2em; text-indent: 1em; }\n"
    "p.lidialog { margin-right: 4em; margin-left: 0em; text-indent: 1em; }\n"
    "p.ridialog { margin-left: 4em; margin-right: 0em; text-indent: 1em; }\n"

    "p.lyric { margin-left: 3em; margin-right: 2em; text-indent: -1em; font-style: italic; }\n"
    "p.llyric { margin-right: 4em; margin-left: 1em; text-indent: -1em; font-style: italic; }\n"
    "p.rlyric { margin-left: 5em; margin-right: 0em; text-indent: -1em; font-style: italic; }\n"

    "p.paren { margin-left: 3em; margin-right: 2em; }\n"
    "p.lparen { margin-right: 2em; margin-left: 1em; }\n"
    "p.rparen { margin-left: 5em; margin-right: 2em; }\n"

    "p.transition { margin-top: 1em; text-align: right; }\n"
    "em { font-style: italic; }\n"
    "p.lyric em, p.llyric em, p.rlyric em { font-weight: 900; }\n";

   FILE *file;

   remove_ebook( root );
   snprintf( buffer, sizeof( buffer ), "%s/ebook", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/META-INF", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS", root );
   make_dir( buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/mimetype", root );
   file = open_file( buffer );
   put_string( mime_type, file );
   fclose( file );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/META-INF/container.xml", root );
   file = open_file( buffer );
   put_string( content_xml, file );
   fclose( file );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/stylesheet.css", root );
   file = open_file( buffer );
   put_string( styles, file );
   fclose( file );
}

char *epub_make_date()
{
   struct tm lt;
   time_t t;
   static char date[ 64 ];

   t = time( NULL );
   localtime_r( &t, &lt );
   strftime( date, sizeof( date ), "%Y-%m-%dT%H:%M:%SZ", &lt );

   return date;
}

char *make_identifier( struct meta *meta )
{
   char *title, *author, *ptr;
   struct string *id;

   if (( id = string_make( NULL )) == NULL )
      return NULL;

   title = meta_escape( meta->title );
   author = meta_escape( meta->author );

   for( ptr = title; *ptr; ++ptr )
      if ( *ptr == ' ' )
         *ptr = '-';

   for( ptr = author; *ptr; ++ptr )
      if ( *ptr == ' ' )
         *ptr = '-';

   concat( id, title );
   concat( id, "-" );
   concat( id, author );
   concat( id, "-" );

   concat( id, epub_make_date());
   free( title );
   free( author );

   ptr = id->str;
   free( id );
   return ptr;
}

void write_toc( char *root, struct string *toc )
{
   FILE *file;
   char buffer[ MAXPATHLEN + 1 ],
        *preamble = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
                    "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:epub=\"http://www.idpf.org/2007/ops\">\n"
                    "<head>\n"
                    "<title>toc.xhtml</title>\n"
                    "<link href=\"stylesheet.css\" rel=\"stylesheet\" type=\"text/css\" />\n"
                    "</head>\n"
                    "<body>\n"
                    "<nav id=\"toc\" epub:type=\"toc\">\n"
                    "<h1>Contents</h1>\n"
                    "<ol>\n";

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/toc.xhtml", root );
   file = open_file( buffer );

   put_string( preamble, file );
   put_string( toc->str, file );
   put_string( "</ol>\n</nav>\n</body>\n</html>\n", file );

   fclose( file );
}

void add_preamble( FILE *file, int fileno )
{
   char buffer[ 1024 ],
        *preamble = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n"
                    "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:epub=\"http://www.idpf.org/2007/ops\">\n"
                    "<head>\n"
                    "<title>file%d.xhtml</title>\n"
                    "<link href=\"stylesheet.css\" rel=\"stylesheet\" type=\"text/css\" />\n"
                    "</head>\n"
                    "<body>\n";

   snprintf( buffer, sizeof( buffer ), preamble, fileno );
   put_string( buffer, file );
}

void add_closer( FILE *file )
{
   put_string( "</body>\n</html>\n", file );
}

void advance_file( FILE **file, int *fileno, char *root, struct string *data, struct string *spine, struct string *toc )
{
   char buffer[ 1024 ];

   if ( *file != NULL )
   {
      ++*fileno;
      add_closer( *file );
      fclose( *file );
   }

   if ( toc != NULL )
   {
      snprintf( buffer, sizeof( buffer ), "<li><a href=\"file%d.xhtml\">", *fileno );
      concat( toc, buffer );
   }

   snprintf( buffer, sizeof( buffer ), "<itemref idref=\"item%d\" />\n", *fileno );
   concat( spine, buffer );

   snprintf( buffer, sizeof( buffer ),
             "<item id=\"item%d\" href=\"file%d.xhtml\" media-type=\"application/xhtml+xml\" />\n",
             *fileno, *fileno );
   concat( data, buffer );

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/file%d.xhtml", root, *fileno );
   *file = open_file( buffer );
   add_preamble( *file, *fileno );
}

void epub_add_title_page( struct meta *meta, FILE **file, int *fileno, char *root,
                          struct string *data, struct string *spine, struct string *toc )
{
   char *tptr = NULL, *aptr = NULL;
   struct string *item;

   advance_file( file, fileno, root, data, spine, NULL );

   if (( item = string_make( NULL )) == NULL )
      exit( 1 );

   concat( item, "<h1>" );
   concat( item, tptr = meta_escape( meta->title ));
   concat( item, "</h1>\n<h2>" );
   concat( item, aptr = meta_escape( meta->author ));
   concat( item, "</h2>\n" );

   free( tptr );
   free( aptr );
   put_string( item->str, *file );

   if ( meta->copyright == NULL )
      goto END;

   string_clear( item );
   concat( item, "<p class=\"copy\">Copyright &#169; " );
   tptr = meta_escape( meta->copyright );
   concat( item, tptr );
   free( tptr );
   concat( item, "</p>\n" );

   put_string( item->str, *file );

END:
   string_free( item);
   advance_file( file, fileno, root, data, spine, toc );
}

void add_files( char *root, struct meta *meta, struct para *paras, struct string *data, int cover )
{
   union stack_val val;
   FILE *file = NULL;
   struct string *toc, *spine;
   struct para *ptr;
   int last_cue_type = -1, fileno = 0, no_first_header = 0;

   if (( toc = string_make( NULL )) == NULL || ( spine = string_make( NULL )) == NULL )
      exit( 1 );

   if ( cover )
   {
      advance_file( &file, &fileno, root, data, spine, NULL );
      put_string( "<p class=\"break\"><img src=\"cover2.png\" alt=\"Cover Image\" /></p>\n", file );
   }

   epub_add_title_page( meta, &file, &fileno, root, data, spine, toc );

   for( ptr = paras; ptr != NULL; ptr = ptr->next )
   {
      switch( ptr->type )
      {
         case ACT:
         case END:
         case SCENE:
         case SHOT:
            if ( ptr != paras )
            {
               if ( no_first_header )
               {
                  concat( toc, "PREFACE</a></li>\n" );
                  no_first_header = 0;
               }

               if ( ptr->type != SHOT )
                  advance_file( &file, &fileno, root, data, spine, toc );
            }

            if ( ptr->type == SCENE || ptr->type == SHOT )
               put_string( "<p class=\"scene\">\n", file );
            else
               put_string( "<h1>", file );

            while( ptr->lines->used )
            {
               stack_shift( ptr->lines, &val );
               put_string( val.ptr, file );
               put_string( " ", file );

               if ( ptr->type != SHOT )
                  concat( toc, val.ptr );

               free( val.ptr );
            }

            if ( ptr->type != SHOT )
               concat( toc, "</a></li>\n" );

            put_string((( ptr->type == SCENE || ptr->type == SHOT ) ? "</p>\n" : "</h1>\n" ), file );
            continue;

         case TRANS:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"transition\">\n", file );
            break;

         case ACTION:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"action\">\n", file );
            break;

         case CUE:
         case LCUE:
         case RCUE:
            last_cue_type = ptr->type;

            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );
            put_string(( last_cue_type == CUE ? "cue\">\n" : ( last_cue_type == LCUE ? "lcue\">\n" : "rcue\">\n" )), file );
            break;

         case DIALOG:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );
            put_string(( last_cue_type == CUE ? "dialog\">\n" : ( last_cue_type == LCUE ? "ldialog\">\n" : "rdialog\">\n" )), file );
            break;

         case IDIALOG:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );
            put_string(( last_cue_type == CUE ? "idialog\">\n" : ( last_cue_type == LCUE ? "lidialog\">\n" : "ridialog\">\n" )), file );
            break;

         case PAREN:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );
            put_string(( last_cue_type == CUE ? "paren\">\n" : ( last_cue_type == LCUE ? "lparen\">\n" : "rparen\">\n" )), file );
            break;

         case LYRIC:
            if ( ptr == paras )
               ++no_first_header;

            put_string( "<p class=\"", file );
            put_string(( last_cue_type == CUE ? "lyric\">\n" : ( last_cue_type == LCUE ? "llyric\">\n" : "rlyric\">\n" )), file );
      }

      while( ptr->lines->used )
      {
         stack_shift( ptr->lines, &val );
         put_string( val.ptr, file );
         put_string( "\n", file );
         free( val.ptr );
      }

      put_string( "</p>\n", file );
   }

   add_closer( file );
   fclose( file );

   if ( no_first_header )
      concat( toc, "PREFACE</a></li>\n" );

   write_toc( root, toc );
   string_free( toc );

   concat( data, "</manifest>\n<spine>\n" );
   concat( data, spine->str );
   concat( data, "</spine>\n" );

   string_free( spine );
}

void add_content( char *root, struct meta *meta, struct para *paras, int cover )
{
   FILE *file;
   struct string *data;
   char buffer[ MAXPATHLEN + 1 ], *ptr,

        *preamble = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
                    "<package xmlns=\"http://www.idpf.org/2007/opf\" "
                             "xmlns:dc=\"http://purl.org/dc/elements/1.1/\" "
                             "unique-identifier=\"book-id\" "
                             "version=\"3.0\">\n"
                    "<metadata>\n",

        *manifest = "<manifest>\n"
                    "<item id=\"toc\" properties=\"nav\" href=\"toc.xhtml\" media-type=\"application/xhtml+xml\" />\n"
                    "<item id=\"css\" href=\"stylesheet.css\" media-type=\"text/css\" />\n";

   snprintf( buffer, sizeof( buffer ), "%s/ebook/OEBPS/content.opf", root );

   file = open_file( buffer );
   put_string( preamble, file );

   if (( data = string_make( NULL )) == NULL )
      exit( 1 );

   concat( data, "<dc:title>" );
   ptr = meta_escape( meta->title );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:title>\n" );

   concat( data, "<dc:creator>" );
   ptr = meta_escape( meta->author );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:creator>\n");

   concat( data, "<dc:identifier id=\"book-id\">" );
   ptr = make_identifier( meta );
   concat( data, ptr );
   free( ptr );
   concat( data, "</dc:identifier>\n" );

   snprintf( buffer, sizeof( buffer ), "<meta property=\"dcterms:modified\">%s</meta>\n",
             epub_make_date());
   concat( data, buffer );

   concat( data, "<dc:language>" );
   concat( data, ( meta->language == NULL ? "en" : meta->language ));
   concat( data, "</dc:language>\n</metadata>\n" );

   concat( data, manifest );

   if ( cover )
   {
      concat( data, "<item id=\"cover1\" properties=\"cover-image\" href=\"cover1.png\" media-type=\"image/png\" />\n" );
      concat( data, "<item id=\"cover2\" href=\"cover2.png\" media-type=\"image/png\" />\n" );
   }

   add_files( root, meta, paras, data, cover );
   concat( data, "</package>\n" );

   put_string( data->str, file );
   string_free( data );
   fclose( file );
}

void make_ebook( char *scenes, char *root, int numbers, int cover, int addcont )
{
   struct meta *meta;
   struct tsml *tree;
   struct para *paras;

   if (( tree = make_tsml_tree( stdin )) == NULL )
      return;

   if (( meta = make_meta()) == NULL )
   {
      free_tsml_tree( tree );
      return;
   }

   if (( paras = epub_make_para_list( tree, meta, scenes, numbers, addcont )) == NULL )
   {
      free_meta( meta );
      free_tsml_tree( tree );
      return;
   }

   free_tsml_tree( tree );

   make_dirs( root );
   add_content( root, meta, paras, cover );

   free_meta( meta );
   free_para_list( paras );
   return;
}
