/*
 * Nellie Copyright (c) 2017-2021, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <unistd.h>
#include <sys/stat.h>

#include "formatter.h"
#include "markup.h"
#include "epub.h"

int SCENE_ORPH = _SCENE_ORPH, CUE_ORPH = _CUE_ORPH, LINES = _LINES, CHARS = _CHARS, 
    FULL_LEN = _FULL_LEN, QUARTER = _QUARTER, HALF = _HALF, TOP = _TOP, WIDTH = _WIDTH,
    LENGTH = _LENGTH, PAGENO = _PAGENO, RCUE_OFF = _RCUE_OFF, RPAREN_OFF = _RPAREN_OFF,
    RDIALOG_OFF = _RDIALOG_OFF;

void *memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
      fprintf( stderr, "malloc(): %s", strerror( errno ));

   return ptr;
}

char *str_dup( char *str, int len )
{
   char *ptr, *ptr2;

   if ( str == NULL )
      return NULL;

   if ( len < 0 )
      for( len = 0, ptr = str; *ptr; ++ptr )
         ++len;

   if (( ptr = memory( len + 1 )) == NULL )
      return NULL;

   ptr[ len ] = '\0';

   for( ptr2 = ptr; len; --len )
      *ptr2++ = *str++;

   return ptr;
}

void print_pdf( char *scenes, int numbers, int addcont, int boldacts, int boldscenes, int boldshots )
{
   struct meta *meta;
   struct tsml *tree;
   struct para *paras;
   struct stack *pages, *pdf;
   int n;

   if (( tree = make_tsml_tree( stdin )) == NULL )
      return;

   if (( meta = make_meta()) == NULL )
   {
      free_tsml_tree( tree );
      return;
   }

   if (( paras = make_para_list( tree, meta, scenes, numbers, addcont )) == NULL )
   {
      free_meta( meta );
      free_tsml_tree( tree );
      return;
   }

   free_tsml_tree( tree );

   if (( pages = make_page_stack( paras, numbers, boldacts, boldscenes, boldshots )) == NULL )
   {
      free_meta( meta );
      free_para_list( paras );
      return;
   }

   free_para_list( paras );

   if (( pdf = make_pdf( pages, meta, boldacts )) == NULL )
   {
      free_meta( meta );
      free_page_stack( pages );
      return;
   }

   free_meta( meta );
   free_page_stack( pages );

   for( n = 0; n < pdf->used; n += 2 )
   {
      fwrite( pdf->values[ n + 1 ].ptr, 1, pdf->values[ n ].num, stdout );
      free( pdf->values[ n + 1 ].ptr );
   }

   stack_free( pdf );
}

void redirect_stdin( char *infile )
{
   FILE *file;

   if ( infile == NULL || ( *infile == '-' && ! infile[ 1 ] ))
      return;

   if (( file = fopen( infile, "r" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"r\" ): %s", infile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdin );
   stdin = file;
}

void redirect_stdout( char *outfile )
{
   FILE *out;

   if ( outfile == NULL )
      return;

   unlink( outfile );
   if (( out = fopen( outfile, "w" )) == NULL )
   {
      fprintf( stderr, "fopen( %s, \"w\" ): %s\n", outfile, strerror( errno ));
      exit( 1 );
   }

   fclose( stdout );
   stdout = out;
}

void set_constants( int a4 )
{
   if ( a4 )
   {
      LENGTH     = _A4_LENGTH;
      PAGENO     = _A4_PAGENO;
      WIDTH      = _A4_WIDTH;
      SCENE_ORPH = _A4_SCENE_ORPH;
      CUE_ORPH   = _A4_CUE_ORPH;
      LINES      = _A4_LINES;
      CHARS      = _A4_CHARS;
      FULL_LEN   = _A4_FULL_LEN;
      QUARTER    = _A4_QUARTER;
      HALF       = _A4_HALF;
      TOP        = _A4_TOP;

      RCUE_OFF    = _A4_RCUE_OFF;
      RPAREN_OFF  = _A4_RPAREN_OFF;
      RDIALOG_OFF = _A4_RDIALOG_OFF;
   }
}

int main( int argc, char **argv )
{
   char *scenes = NULL, *dir = ".";
   int opt, args = 0, numbers = 0, convert = 0, epub = 0,
            a4 = 0, cover = 0, addcont = 1, boldacts = 1, boldscenes = 1, boldshots = 1;

   while(( opt = getopt( argc, argv, "btoisacneid:" )) != -1 )
   {
      ++args;

      switch( opt )
      {
         case 'a':
            ++a4;
            break;

         case 'b':
            --boldscenes;
            break;

         case 'c':
            ++convert;
            break;

         case 'd':
            scenes = optarg;
            ++args;
            break;

         case 'e':
            ++epub;
            break;
         
         case 'i':
            ++cover;
            break;
         
         case 'n':
            ++numbers;
            break;

         case 'o':
            --boldacts;
            break;

         case 's':
            --addcont;
            break;

         case 't':
            --boldshots;
            break;
      }         
   }

   opt = convert + epub;

   if ( opt > 1 )
   {
      fputs( "-c and -e are mutually exclusive\n", stderr );
      exit( 1 );
   }

   argc -= args;
   argv += args;

   if ( argc > 1 )
      redirect_stdin( argv[ 1 ] );

   if ( argc > 2 )
   {
      if ( epub )
         dir = argv[ 2 ];
      else
         redirect_stdout( argv[ 2 ] );
   }

   set_constants( a4 );
   
   if ( convert )
      print_tsml();
   else if ( epub )
      make_ebook( scenes, dir, numbers, cover, addcont );
   else
      print_pdf( scenes, numbers, addcont, boldacts, boldscenes, boldshots );

   return 0;
}
