/*
 * Nellie Copyright (c) 2017-2021, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "formatter.h"

extern char *str_dup( char *, int );

int get_line( struct string *item, int spaces )
{
   int c;

   string_clear( item );

   while(( c = fgetc( stdin )) != EOF )
   {
      if ( c == '\r' )
         continue;

      if ( c == '\n' )
         break;

      if (( c == '[' || c == ']' || c == '\\' ) && string_append( item, '\\' ))
         exit( 1 );

      if ( isspace( c ))
         c = ' ';

      if ( string_append( item, ( char )c ))
         exit( 1 );
   }

   if ( c == EOF && ! item->used )
      return 1;

   if ( ! spaces )
      while( item->used && item->str[ item->used - 1 ] == ' ' )
      {
         --item->used;
         ++item->free;
         --item->top;
         *item->top = '\0';
      }

   return 0;
}

int compare( char *ptr1, char *ptr2 )
{
   char *ptr;

   for( ptr = ptr2; *ptr; ++ptr )
      *ptr = tolower( *ptr );

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      return 1;

   return 0;
}

int is_blank( char *item )
{
   char *ptr;

   for( ptr = item; *ptr; ++ptr )
      if ( ! isspace( *ptr ))
         return 0;

   return 1;
}

void replace_semicolons( char *item )
{
   char *ptr;

   for( ptr = item; *ptr; ++ptr )
      if ( *ptr == ';' )
         *ptr = '~';
}

int print_title_page( struct string *item )
{
   char *ptr;
   int first = 1, multi, found = 0;

   fputs( "[m[\n", stdout );

   for( multi = 0; ; )
   {
      if ( get_line( item, 1 ))
         break;

      if ( !item->used )
      {
         if ( first )
            continue;
         else
            break;
      }

      first = 0;

      if ( multi )
      {
         if ( isspace( *item->str ))
         {
            replace_semicolons( item->str );
            printf( "%s%s", ( multi > 1 ? ";" : "" ), item->str );
            ++multi;
            continue;
         }
         else
         {
            fputs( "]]\n", stdout );
            multi = 0;
         }
      }

      for( ptr = item->str; *ptr; ++ptr )
         if ( *ptr == ':' )
         {
            ++found;
            *ptr = '\0';

            if ( compare( "title",      item->str ) || compare( "author",  item->str )   ||
                 compare( "draft date", item->str ) || compare( "contact", item->str )   ||
                 compare( "watermark",  item->str ) || compare( "copyright", item->str ) ||
                 compare( "language",   item->str ))
            {
               char c;

               switch( *item->str )
               {
                  case 't':
                     c = 't';
                     break;

                  case 'a':
                     c = 'a';
                     break;

                  case 'd':
                     c = 'r';
                     break;

                  case 'c':
                     if ( item->str[ 2 ] == 'p' )
                        c = 'y';
                     else
                        c = 'c';
                     break;

                  case 'l':
                     c = 'l';
                     break;

                  case 'w':
                     c = 'w';
                     break;

                  default:
                     c = '\0';
               }

               if ( ! is_blank( ++ptr ))
               {
                  replace_semicolons( ptr );
                  printf( "   [%c[%s]]\n", c, ptr );
               }
               else
               {
                  printf( "   [%c[", c );
                  ++multi;
               }
            }

            break;
         }

      if ( ! found )
         break;
   }

   if ( multi )
      fputs( "]]\n", stdout );

   fputs( "]m]\n\n", stdout );
   return found;
}

int is_cue( struct string *item )
{
   char *ptr;

   if ( ! item->used || *item->str == '!' )
      return 0;

   if ( *item->str == '@' || *item->str == '-' || *item->str == '+' )
   {
      char type = *item->str;

      for( ptr = item->str; *ptr; ++ptr )
         *ptr = *( ptr + 1 );

      --item->used;
      ++item->free;
      --item->top;

      if ( type == '@' )
         return CUE;
      else if ( type == '-' )
         return LCUE;
      else
         return RCUE;
   }

   for( ptr = item->str; *ptr; ++ptr )
      if ( *ptr != ' ' && ! isdigit( *ptr ) && ( ! isalpha( *ptr ) || ! isupper( *ptr )))
         return 0;

   return CUE;
}

int is_paren( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( *item->str == '(' && item->str[ item->used - 1 ] == ')' )
      return 1;

   return 0;
}

int is_trans( struct string *item )
{
   char *ptr;

   if ( ! item->used || *item->str == '!' )
      return 0;

   if ( *item->str == '>' && item->str[ item->used - 1 ] != '<' )
   {
      for( ptr = item->str; *ptr; ++ptr )
         *ptr = *( ptr + 1 );

      --item->used;
      ++item->free;
      --item->top;
      return 1;
   }

   for( ptr = item->str; *ptr; ++ptr )
      if ( *ptr != ' ' && *ptr != ':' && ! isdigit( *ptr ) && ( ! isalpha( *ptr ) || ! isupper( *ptr )))
         return 0;

   if ( item->used > 3 && item->str[ item->used - 1 ] == ':' &&
                          item->str[ item->used - 2 ] == 'O' &&
                          item->str[ item->used - 3 ] == 'T' &&
                          item->str[ item->used - 4 ] == ' ' )
      return 1;

   return 0;
}

int is_act( struct string *item )
{
   char *ptr;

   if ( ! item->used || *item->str != '>' || item->str[ item->used - 1 ] != '<' )
      return 0;

   item->str[ item->used - 1 ] = '\0';

   --item->used;
   ++item->free;
   --item->top;

   for( ptr = item->str; *ptr; ++ptr )
      *ptr = *( ptr + 1 );

   --item->used;
   ++item->free;
   --item->top;

   return 1;
}

int is_centered( struct string *item )
{
   if ( is_act( item ))
   {
      if ( item->used > 2 && item->str[ 0 ] == 'E' &&
                             item->str[ 1 ] == 'N' &&
                             item->str[ 2 ] == 'D' )
         return 2;

      return 1;
   }

   return 0;
}

void remove_scene_number_upcase( struct string *item )
{
   char *ptr;

   ptr = &item->str[ item->used - 1 ];

   if ( *ptr == '#' )
   {
      do
      {
         *ptr-- = '\0';

         --item->top;
         ++item->free;
         --item->used;
      }
      while ( item->used && ( *ptr == '#' || *ptr == ' ' || isdigit( *ptr )));
   }

   for( ptr = item->str; *ptr; ++ptr )
      *ptr = toupper( *ptr );
}

int is_shot( struct string *item )
{
   char *ptr;

   if ( ! item->used || *item->str == '!' )
      return 0;

   if ( *item->str == ':' )
   {
      for( ptr = item->str; *ptr; ++ptr )
         *ptr = *( ptr + 1 );

      --item->used;
      ++item->free;
      --item->top;

      for( ptr = item->str; *ptr; ++ptr )
         *ptr = toupper( *ptr );

      return 1;
   }

   return 0;
}

int is_scene( struct string *item )
{
   char *ptr;

   if ( ! item->used || *item->str == '!' )
      return 0;

   if ( *item->str == '.' )
   {
      for( ptr = item->str; *ptr; ++ptr )
         *ptr = *( ptr + 1 );

      --item->used;
      ++item->free;
      --item->top;

      remove_scene_number_upcase( item );
      return 1;
   }

   if (((( item->str[ 0 ] == 'I' || item->str[ 0 ] == 'i' ) && ( item->str[ 1 ] == 'N' || item->str[ 1 ] == 'n' )) ||
        (( item->str[ 0 ] == 'E' || item->str[ 0 ] == 'e' ) && ( item->str[ 1 ] == 'X' || item->str[ 1 ] == 'x' ))) &&
       ( item->str[ 2 ] == 'T' || item->str[ 2 ] == 't' ) && item->str[ 3 ] == '.' )
   {
      remove_scene_number_upcase( item );
      return 1;
   }

   return 0;
}

int is_page_break( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->used == 3 && item->str[ 0 ] == '=' && item->str[ 1 ] == '=' && item->str[ 2 ] == '=' )
      return 1;

   return 0;
}

int is_note( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->used > 3 && item->str[ 0 ] == '\\' && item->str[ 1 ] == '['
                       && item->str[ 2 ] == '\\' && item->str[ 3 ] == '[' )
   {
      do
         if ( item->used > 3 && item->str[ item->used - 1 ] == ']' && item->str[ item->used - 2 ] == '\\'
                             && item->str[ item->used - 3 ] == ']' && item->str[ item->used - 4 ] == '\\' )
            break;
      while( ! get_line( item, 0 ));

      return 1;
   }

   return 0;
}

/*
 * Converts a paragraph containing only a metacharacter into a paragraph of
 * a single space to enable the creation of blank lines in dialog.
 */

void remove_character( struct string *item, char c )
{
   if ( item->used && *item->str == c )
   {
      char *ptr;

      if ( item->used == 1 && c != '!' )
      {
         *item->str = ' ' ;
         return;
      }

      if (( ptr = str_dup( &item->str[ 1 ], item->used - 1 )) == NULL )
         return;

      free( item->str );
      item->str = ptr;

      --item->used;
      item->free = 0;
      item->top = &item->str[ item->used - 1 ];
   }

   return;
}

int is_indented( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->str[ 0 ] == '&' )
      return 1;

   return 0;
}

int is_outdented( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->str[ 0 ] == '$' )
      return 1;

   return 0;
}

int is_lyric( struct string *item )
{
   if ( ! item->used )
      return 0;

   if ( item->str[ 0 ] == '~' )
      return 1;

   return 0;
}

char dialog_type( struct string *para, char last, int clean )
{
   char c = 0;

   if ( last != 'c' && last != 'u' && last != 'f' && last != 'r' && last != 'p' && last != 'i' && last != 'l' && last != 'd' )
      return -1;

   if ( is_paren( para ))
      c = 'p';
   else if ( is_indented( para ))
   {
      c = 'i';

      if ( clean )
         remove_character( para, '&' );
   }
   else if ( is_outdented( para ))
   {
      c = 'd';

      if ( clean )
         remove_character( para, '$' );
   }
   else if ( is_lyric( para ))
   {
      c = 'l';

      if ( clean )
         remove_character( para, '~' );
   }

   return c;
}

void print_action_or_dialog( struct string *para, char last, int clean )
{
   char c;

   if ( ! para->used )
      return;

   if (( c = dialog_type( para, last, clean )) < 0 )
   {
      c = 'a';
      remove_character( para, '!' );
   }

   printf( "[%c[%s]]\n\n", ( c ? c : 'd' ), para->str );
   string_clear( para );
}

void print_scenes( struct string *item, int found )
{
   struct string *para;
   int c, l = 0, eof;

   if (( para = string_make( NULL )) == NULL )
      return;

   for( eof = ( found ? get_line( item, 0 ) : 0 ); ! eof; eof = get_line( item, 0 ))
   {
      if ( is_page_break( item ))
         continue;

      if ( para->used )
         c = dialog_type( item, l, 0 );
      else
      {
         if (( c = is_centered( item )))
            c = ( c == 2 ? 'e' : 'v' );
         else if ( is_note( item ))
            continue;
         else if ( is_scene( item ))
            c = 's';
         else if ( is_shot( item ))
            c = 'o';
         else if ( is_trans( item ))
            c = 't';
         else if (( c = is_cue( item )))
            c = ( c == CUE ? 'c' : ( c == LCUE ? 'f' : 'r' ));
         else
            c = dialog_type( item, l, 0 );
      }

      if ( c > 0 )
      {
         if ( para->used && ( c == 'p' || c == 'i' || c == 'l' || c == 'd' ))
         {
            char z = dialog_type( para, l, 1 );
            printf( "[%c[%s]]\n", ( z > 0 ? z : 'd' ), para->str );
            string_clear( para );
         }

         if ( c == 'i' || c == 'd' )
            goto COLLECT;
         else if ( c == 'l' )
            remove_character( item, '~' );

         printf( "[%c[%s]]\n", c, item->str );
         l = c;
         continue;
      }

COLLECT:
      if ( item->used )
      {
         if ( para->used && string_append( para, ' ' ))
            break;

         if ( string_merge( para, item ))
            break;

         continue;
      }

      print_action_or_dialog( para, l, 1 );
      l = 0;
   }

   print_action_or_dialog( para, l, 1 );
   string_free( para );
}

void print_tsml()
{
   struct string *item;
   int found;

   if (( item = string_make( NULL )) == NULL )
      exit( 1 );

   fputs( "[y[\n", stdout );

   found = print_title_page( item );
   print_scenes( item, found );

   fputs( "]y]\n", stdout );
   string_free( item );
}
