/*
 * House Copyright (c) 2016-2022, James Bailie.
 * All rights reserved.

 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/un.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

#include <pwd.h>
#include <grp.h>

#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#define _WITH_DPRINTF 1

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <math.h>

#define PIDFILE "/var/run/house.pid"

/*
 * Maximum simultaneous connections.
 */

#define MAX_CONN 4096

/*
 * Maximum number of new events generated for every event processed.
 */

#define MAX_EVENT 10

/*
 * Maximum number of room occupants.
 */

#define MAX_OCCUPANTS 32

/*
 * Maximum length of user names.
 */

#define MAX_NAME 25

/*
 * Interval between timeout checks.
 */

#define INTERVAL 5 * 60

/*
 * Maximum size of chunks of data read and written from/to client.
 * Cannot be more than 65540:  65535 + 4 byte header + 1 byte terminator.
 */

#define IOBUFSIZE 1024

#define HEAD 4

/*
 * Maximum size of connection output queues.

 * With each pass through the inqueue, the other people in your room can
 * generate MAX_OCCUPANTS - 1 messages in your queue.  You queue 1 item if
 * you chat.  If you invoke .who, you generate MAX_OCCUPANTS - 1 messages.
 * If you invoke .help, you queue 2 fragments.

 * Only 1 item is drained from a connection's queue on each write event.
 * We're counting on write events occurring much faster than read events so
 * that the queue is drained quickly into the socket send buffer.  A busy
 * system will drop data when the queue is full.
 */

#define MAX_QUEUE MAX_OCCUPANTS * 2

/*
 * The amount of space reserved in ccb buffers for annotation of chat data
 * with markup and user name.

 * Incoming buffers need room for the 4 byte header, a newline.  Outgoing
 * buffers need room for a 100 byte user name, and some markup.  This gives
 * us room for 199 bytes of markup and a terminator.
 */

#define GAP 300

/*
 * Maximum length of room names.  Over 57 million names are possible.
 */

#define ROOMSIZE 7

/*
 * Times in seconds.
 */

/*
 * Connection idle timeout.
 */

#define SECONDS 10 * 60

/*
 * Default number of new connections accepted at one time in
 * accept_connection(). backlog is now set to NEWCONS.
 */

#define NEWCONS 1024

/*
 * KQueue Input and output queues.  We can add a maximum of 10 events to
 * the input queue for every item in the output queue + NEWCONS for new
 * connections that accept_connection() can generate.
 */

struct kevent *inqueue, *outqueue;

extern char *optarg;

unsigned int max_conn = MAX_CONN, stack_inc = 120, seconds = SECONDS, qlen = MAX_CONN * MAX_EVENT,
             alloc_err = 0, max_occupants = MAX_OCCUPANTS, iobufsize = IOBUFSIZE;

int in = 0, out = 0, fd = -1, logging = 0, testing = 0, hash_size = 16384,
    backlog = 1000, closed = 0, conn_count = 0;

char *grp = "www", *user = "www", *listen_unix = "/var/run/house.socket",
     **g_argv, *hostname = "";

struct passwd *passwd;
struct group *group;

volatile int sigterm = 0;

/*
 * Queue of buffered out-going data for each connection.
 */

struct buffer
{
   int refcount;
   char *buffer;
};

struct queue
{
   int count, total;
   struct buffer *buffer;
   struct queue *next;
};

/*
 * Connection control block.
 */

struct ccb
{
   unsigned int sock, total, stage, len, offset, qlen, flen, namelen, roomlen, closed;
   time_t timestamp;
   struct queue *first, *last;
   char name[ MAX_NAME * 4 + 1 ], *buffer, mask[ 4 ], room[ ROOMSIZE + 1 ];
} *conns = NULL;

struct list
{
   struct list *next;
   struct ccb *conn;
};

struct hash
{
   char *name;
   int lock;

   struct hash *next;
   struct list *conns;
   int count;
} **hash = NULL;

void remove_conn( struct ccb * );
void queue_enter_exit_messages( struct ccb *, int );

void go ( struct ccb *, int, char * );
void who( struct ccb *, int, char * );
void help( struct ccb *, int, char * );

void room( struct ccb *, int, char * );
void leave( struct ccb *, int, char * );

void lock( struct ccb *, int, char * );
void unlock( struct ccb *, int, char * );

struct stack
{
   int free, used;
   struct ccb **top, **values;
} *free_conns = NULL;

void *memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
   {
      if ( logging )
         syslog( LOG_WARNING, "malloc(): %m" );
      else
         fprintf( stderr, "house: malloc(): %s\n", strerror( errno ));

      return NULL;
   }

   return ptr;
}

char *str_dup( char *str, int len )
{
   char *ptr, *ptr2;

   if ( str == NULL )
      return NULL;

   if ( len <= 0 )
      for( len = 0, ptr = str; *ptr; ++ptr )
         ++len;

   if (( ptr = memory( len + 1 )) == NULL )
      return NULL;

   ptr[ len ] = '\0';

   for( ptr2 = ptr; len; --len )
      *ptr2++ = *str++;

   return ptr;
}

int insert_room( struct ccb *item )
{
   unsigned int key;
   char *rptr;
   struct list *lptr;
   struct hash *ptr = NULL, *ptr2 = NULL;

   for( key = 2166136261, rptr = item->room; *rptr; ++rptr )
      key = ( key * 16777619 ) ^ *rptr;

   key %= hash_size;

   if ( hash[ key ] == NULL )
   {
      if (( hash[ key ] = memory( sizeof( struct hash ))) == NULL )
         return 1;

      hash[ key ]->next = NULL;

      if (( hash[ key ]->conns = memory( sizeof( struct list ))) == NULL )
      {
         free( hash[ key ] );
         hash[ key ] = NULL;
         return 1;
      }

      hash[ key ]->conns->next = NULL;
      hash[ key ]->conns->conn = item;

      if (( hash[ key ]->name = str_dup( item->room, item->roomlen )) == NULL )
      {
         free( hash[ key ] );
         hash[ key ] = NULL;
         return 1;
      }

      hash[ key ]->lock = 0;
      hash[ key ]->count = 1;
      return 0;
   }

   ptr2 = hash[ key ];

   for( ptr = ptr2; ptr != NULL; ptr = ptr->next )
   {
      char *ptr3, *ptr4;

      ptr3 = ptr->name;
      ptr4 = item->room;

      while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
      {
         ++ptr3;
         ++ptr4;
      }

      if ( ! *ptr3 && ! *ptr4 )
         break;

      ptr2 = ptr;
   }

   if ( ptr == NULL )
   {
      if (( ptr2->next = memory( sizeof( struct hash ))) == NULL )
         return 1;

      ptr = ptr2->next;
      ptr->next = NULL;

      if (( ptr->conns = memory( sizeof( struct list ))) == NULL )
      {
         free( ptr2->next );
         ptr2->next = NULL;
         return 1;
      }

      ptr->conns->conn = item;
      ptr->conns->next = NULL;
      ptr->count = 1;
      return 0;
   }

   for( lptr = ptr->conns; lptr != NULL; lptr = lptr->next )
      if ( lptr->next == NULL )
         break;

   if (( lptr->next = memory( sizeof( struct list ))) == NULL )
      return 1;

   ++ptr->count;
   lptr = lptr->next;
   lptr->conn = item;
   lptr->next = NULL;

   return 0;
}

void remove_room( struct ccb *item )
{
   unsigned int key;
   char *rptr;
   struct list *lptr, *lptr2;
   struct hash *ptr, *ptr2;

   if ( ! item->roomlen )
      return;

   for( key = 2166136261, rptr = item->room; *rptr; ++rptr )
      key = ( key * 16777619 ) ^ *rptr;

   key %= hash_size;

   if ( hash[ key ] == NULL )
      return;

   ptr = hash[ key ];
   ptr2 = NULL;

   do
   {
      char *ptr3, *ptr4;

      ptr3 = ptr->name;
      ptr4 = item->room;

      while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
      {
         ++ptr3;
         ++ptr4;
      }

      if ( ! *ptr3 && ! *ptr4 )
         break;

      ptr2 = ptr;
      ptr = ptr->next;
   }
   while( ptr != NULL );

   if ( ptr == NULL )
      return;

   lptr2 = NULL;

   for( lptr = ptr->conns; lptr != NULL; lptr = lptr->next )
   {
      if ( lptr->conn == item )
         break;

      lptr2 = lptr;
   }

   if ( lptr == NULL )
      return;

   if ( lptr2 == NULL )
      ptr->conns = lptr->next;
   else
      lptr2->next = lptr->next;

   free( lptr );
   --ptr->count;

   if ( ptr->conns == NULL )
   {
      if ( ptr2 == NULL )
         hash[ key ] = ptr->next;
      else
         ptr2->next = ptr->next;

      free( ptr->name );
      free( ptr );
   }

   return;
}

struct list *lookup_occupants( char *room, int *clients, int *lock, int setlock )
{
   unsigned int key;
   char *rptr;
   struct hash *ptr;

   if ( clients != NULL )
      *clients = 0;

   if ( lock != NULL )
      *lock = 0;

   for( key = 2166136261, rptr = room; *rptr; ++rptr )
      key = ( key * 16777619 ) ^ *rptr;

   key %= hash_size;

   if ( hash[ key ] == NULL )
      return NULL;

   ptr = hash[ key ];

   do
   {
      char *ptr3, *ptr4;

      ptr3 = ptr->name;
      ptr4 = room;

      while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
      {
         ++ptr3;
         ++ptr4;
      }

      if ( ! *ptr3 && ! *ptr4 )
         break;

      ptr = ptr->next;
   }
   while( ptr != NULL );

   if ( ptr == NULL )
      return NULL;

   if ( setlock )
      ptr->lock = ( setlock < 0 ? 0 : 1 );

   if ( clients != NULL )
      *clients = ptr->count;

   if ( lock != NULL )
      *lock = ptr->lock;

   return ptr->conns;
}

struct stack *make_stack()
{
   struct stack *a;

   if (( a = ( struct stack *)memory( sizeof( struct stack ))) == NULL )
      return NULL;

   if (( a->values = memory( sizeof( struct ccb * ) * stack_inc )) == NULL )
   {
      free( a );
      return NULL;
   }

   a->free = stack_inc;
   a->used = 0;
   a->top = a->values;

   return a;
}

void stack_push( struct stack *a, struct ccb *o )
{
   alloc_err = 0;

   if ( a->free == 0 )
   {
      a->values = realloc( a->values, sizeof( struct ccb * ) * ( a->used + stack_inc ) );

      if ( a->values == NULL )
      {
         syslog( LOG_WARNING, "realloc(): %m" );
         ++alloc_err;
         return;
      }

      a->free = stack_inc;
      a->top = &a->values[ a->used - 1 ];
   }

   if ( a->used )
      ++a->top;

   *a->top = o;
   --a->free;
   ++a->used;
}

#define STACK_PUSH( _s_, _o_ ) { if (( _s_ )->free )\
   {\
      if (( _s_ )->used ) ++( _s_ )->top;\
      --( _s_ )->free;\
      ++( _s_ )->used;\
      *( _s_ )->top = ( _o_ );\
   }\
   else stack_push( _s_, _o_ ); }

struct ccb *stack_pop( struct stack *a )
{
   struct ccb *ptr;

   if ( a->used )
   {
      ptr = *a->top;
      --a->used;
      ++a->free;

      if ( a->used )
         --a->top;

      return ptr;
   }

   return NULL;
}

void ev_set( int desc, short filter, u_short flags, struct ccb *item )
{
   struct kevent *kev;

   if ( in >= qlen )
      return;

   kev = &inqueue[ in++ ];

   kev->ident = desc;
   kev->filter = filter;
   kev->fflags = 0;
   kev->flags = flags;
   kev->udata = item;
}

void set_timer()
{
   struct kevent *kev;

   if ( in >= qlen )
      return;

   kev = &inqueue[ in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = INTERVAL * 1000;
   kev->flags = EV_ADD;
   kev->udata = NULL;
}

void non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }
}

void set_options( int argc, char **argv )
{
   int i;

   while(( i = getopt( argc, argv, "xh:l:u:g:t:m:o:" )) != -1 )
      switch( i )
      {
         case 'g':
            grp = optarg;
            break;

         case 'h':
            hostname = optarg;
            break;

         case 'l':
            listen_unix = optarg;
            break;

         case 'm':
            max_conn = strtol( optarg, NULL, 10 );
            break;

         case 'o':
            max_occupants = strtol( optarg, NULL, 10 );
            break;

         case 't':
            seconds = strtol( optarg, NULL, 10 ) * 60;
            break;

         case 'u':
            user = optarg;
            break;

         case 'x':
            ++testing;
            break;
      }

   if ( max_conn <= 0 )
   {
      fprintf( stderr, "house: max_conn <= 0: %d\n", max_conn );
      exit( 1 );
   }

   if ( seconds <= 0 )
   {
      fprintf( stderr, "house: timeout <= 0: %d\n", seconds / 60 );
      exit( 1 );
   }

   if ( max_occupants <= 1 )
   {
      fprintf( stderr, "house: max_occupants <= 1: %d\n", max_occupants );
      exit( 1 );
   }

   if (( passwd = getpwnam( user )) == NULL )
   {
      fprintf( stderr, "house: user \"%s\" does not exist\n", user );
      exit( 1 );
   }

   if (( group = getgrnam( grp )) == NULL )
   {
      fprintf( stderr, "house: group \"%s\" does not exist\n", grp );
      exit( 1 );
   }
}

void become_daemon()
{
   int file;

   /*
    * Fork and let the parent die, continuing as child so we are not
    * a process group leader.  This is necessary for the call to setsid().
    */

   switch( fork() )
   {
      case -1:
         fprintf( stderr, "house: fork(): %s\n", strerror( errno ));
         exit( 1 );

      case 0:
         break;

      default:
         exit( 0 );
   }

   fclose( stdout );
   fclose( stderr );
   fclose( stdin );

   stdin = fopen( "/dev/null", "r" );
   stdout = fopen( "/dev/null", "w" );
   stderr = fopen( "/dev/null", "w" );

   if ( stdin == NULL || stdout == NULL || stderr == NULL )
   {
      syslog( LOG_ERR, "fopen(): %m" );
      exit( 1 );
   }

   /*
    * Detach us from our controlling terminal, so job control and other
    * signals may not be sent to us from that terminal.
    */

   if ( setsid() < 0 )
   {
      syslog( LOG_ERR, "setsid(): %m" );
      exit( 1 );
   }

   /*
    * Write our pid to disk.
    */

   if (( file = open( PIDFILE, O_WRONLY | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP )) < 0 )
      syslog( LOG_WARNING, "open(): %m" );
   else
   {
      char buffer[ 16 ];
      int t;

      if (( t = snprintf( buffer, sizeof( buffer ), "%d", getpid())) < 0 )
      {
         syslog( LOG_ERR, "snprintf(): %m ");
         exit( 1 );
      }

      write( file, buffer, t );
      close( file );
   }

   /*
    * We don't create any files or fork children which create files,
    * but if we did, this will make those files not world-readable by
    * default.
    */

   umask( 2 );
}

void change_identity()
{
   if ( setgid( group->gr_gid ) < 0 )
      syslog( LOG_ERR, "setgid(): %m" );

   if ( setuid( passwd->pw_uid ) < 0 )
      syslog( LOG_ERR, "setuid(): %m" );
}

void start_listening_unix()
{
   struct sockaddr_un sa;

   if (( fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   unlink( listen_unix );
   bzero( &sa, sizeof( struct sockaddr_un ));
   sa.sun_family = AF_UNIX;
   strncpy( sa.sun_path, listen_unix, sizeof( sa.sun_path ) - 1 );  /* ensures NUL-terminated. */

   if ( bind( fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )))
   {
      syslog( LOG_ERR, "bind( %s ): %m", listen_unix );
      close( fd );
      exit( 1 );
   }

   if ( chown( listen_unix, passwd->pw_uid, group->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "chown( %s ): %m", listen_unix );
      close( fd );
      exit( 1 );
   }

   if ( chmod( listen_unix, S_IRWXU | S_IRWXG ) < 0 )
   {
      syslog( LOG_ERR, "chmod( %s, S_IRWXU | S_IRWXG ): %m", listen_unix );
      close( fd );
      exit( 1 );
   }

   if ( listen( fd, backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      close( fd );
      exit( 1 );
   }

   non_blocking( fd );
}

int queue_pong( struct ccb *item )
{
   struct queue *q;

   if ( item->qlen >= MAX_QUEUE )
      return 1;

   if (( q = memory( sizeof( struct queue ))) == NULL )
      return 1;

   if (( q->buffer = memory( sizeof( struct buffer ))) == NULL )
   {
      free( q );
      return 1;
   }

   q->buffer->refcount = 1;

   if (( q->buffer->buffer = memory( item->len + item->offset )) == NULL )
   {
      free( q->buffer );
      free( q );
      return 1;
   }

   ++item->qlen;

   if ( item->first == NULL )
   {
      item->first = item->last = q;
      ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
   }
   else
   {
      item->last->next = q;
      item->last = q;
   }

   item->last->next = NULL;
   item->last->count = 0;
   item->last->total = item->len + item->offset;

   item->last->buffer->buffer[ 0 ] = 0x8a;

   if ( item->offset == 2 )
      item->last->buffer->buffer[ 1 ] = item->len;
   else
   {
      item->last->buffer->buffer[ 1 ] = 126;
      item->last->buffer->buffer[ 2 ] = item->buffer[ 2 ];
      item->last->buffer->buffer[ 3 ] = item->buffer[ 3 ];
   }

   bcopy( &item->buffer[ item->offset ], &item->last->buffer->buffer[ item->offset ], item->len );
   return 0;
}

/*
 * If item is non-null and others is null, we queue data for item.

 * If others is non-null, we queue data for the others.
 * If item is non-null, we skip over it if it is in the others.
 */

int queue_each_client( struct ccb *item, struct list *others, struct buffer *buffer, int len )
{
   struct list *list;
   struct queue *q;

   if ( others == NULL && item != NULL )
   {
      if ( item->closed || item->qlen >= MAX_QUEUE )
         return 0;

      if (( q = memory( sizeof( struct queue ))) == NULL )
         return 1;

      q->buffer = buffer;
      ++q->buffer->refcount;
      ++item->qlen;

      if ( item->first == NULL )
      {
         item->first = item->last = q;
         ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
      }
      else
      {
         item->last->next = q;
         item->last = q;
      }

      item->last->next = NULL;
      item->last->count = 0;
      item->last->total = len;

      return 0;
   }

   for( list = others; list != NULL; list = list->next )
   {
      if ( list->conn == item || list->conn->qlen >= MAX_QUEUE || list->conn->closed )
         continue;

      if (( q = memory( sizeof( struct queue ))) == NULL )
         return 1;

      q->buffer = buffer;
      ++q->buffer->refcount;
      ++list->conn->qlen;

      if ( list->conn->first == NULL )
      {
         list->conn->first = list->conn->last = q;
         ev_set( list->conn->sock, EVFILT_WRITE, EV_ENABLE, list->conn );
      }
      else
      {
         list->conn->last->next = q;
         list->conn->last = q;
      }

      list->conn->last->next = NULL;
      list->conn->last->count = 0;
      list->conn->last->total = len;
   }

   return 0;
}

/*
 * It is legal to call this function with item == NULL.
 */

int queue_data( struct ccb *item, struct list *others, char *data, int len )
{
   struct buffer *buffer;
   int i, flen, frames;
   char *payload;

   if ( len < 1 )
      return -2;

   if ( ! ( frames = len / ( iobufsize - HEAD )) || len % ( iobufsize - HEAD ))
      ++frames;

   payload = data;

   for( i = 0; i < frames; ++i )
   {
      if (( buffer = memory( sizeof( struct buffer ))) == NULL )
         return -1;

      flen = ( len < ( iobufsize - HEAD ) ? len + 4 : iobufsize );

      if (( buffer->buffer = memory( flen )) == NULL )
      {
         free( buffer );
         return -1;
      }

      buffer->refcount = 0;

      if ( ! i )
         buffer->buffer[ 0 ] = ( frames == 1 ? 0x81 : 0x01 );
      else
         buffer->buffer[ 0 ] = ( i == ( frames - 1 ) ? 0x80 : 0x00 );

      flen = len;

      if ( flen < 126 )
      {
         buffer->buffer[ 1 ] = flen;
         bcopy( payload, &buffer->buffer[ 2 ], flen );
      }
      else
      {
         if ( i != ( frames - 1 ))
         {
            flen = iobufsize - HEAD;
            len -= flen;
         }

         buffer->buffer[ 1 ] = 126;
         buffer->buffer[ 2 ] = flen / 256;
         buffer->buffer[ 3 ] = flen % 256;

         bcopy( payload, &buffer->buffer[ 4 ], flen );
         payload += flen;
      }

      buffer->refcount = 0;

      if ( queue_each_client( item, others, buffer, ( flen < 126 ? flen + 2 : flen + 4 )))
      {
         free( buffer->buffer );
         free( buffer );
         return -1;
      }
   }

   return 0;
}

int utf8_strlen( char *str, int *true_len )
{
   int len = 0;

   *true_len = 0;

   while( *str )
   {
      if (( *str & 0xc0 ) != 0x80 )
         ++len;

      ++*true_len;
      ++str;
   }

   return len;
}

int check_name( struct ccb *item )
{
   char *ptr,
        *try_msg   = "Try another name.",
        *long_msg  = "That name is too long.",
        *blnk_msg  = "That name is blank.",
        *bad_msg   = "That name has illegal characters.";

   int tru_len;
   static int long_len = 0, try_len = 0, bad_len = 0, blnk_len = 0;

   if ( ! long_len )
   {
      ptr = long_msg;

      while( *ptr++ )
         ++long_len;

      ptr = try_msg;

      while( *ptr++ )
         ++try_len;

      ptr = bad_msg;

      while( *ptr++ )
         ++bad_len;

      ptr = blnk_msg;

      while( *ptr++ )
         ++blnk_len;
   }

   /*
    * Trim leading whitespace.
    */

   for( ptr = &item->buffer[ item->offset ]; *ptr == ' ' || *ptr == '\t'; )
   {
      char *ptr1, *ptr2;

      ptr1 = &item->buffer[ item->offset ];
      ptr2 = ptr + 1;

      while( *ptr2 )
         *ptr1++ = *ptr2++;

      *ptr1 = '\0';
   }

   /*
    * Reject bad chars.
    */

   for( ptr = &item->buffer[ item->offset ]; *ptr; ++ptr )
      if ( *ptr == '.' || *ptr == '\'' || *ptr == '"' )
      {
         if ( queue_data( item, NULL, bad_msg, bad_len ) || queue_data( item, NULL, try_msg, try_len ))
            remove_conn( item );

         return 1;
      }

   /*
    * Trim trailing whitespace.
    */

   while( --ptr >= &item->buffer[ item->offset ] && ( *ptr == ' ' || *ptr == '\t' ))
      *ptr = '\0';

   /*
    * Reject bad length.
    */

   if ( ! item->buffer[ item->offset ] )
   {
      if ( queue_data( item, NULL, blnk_msg, blnk_len ) || queue_data( item, NULL, try_msg, try_len ))
         remove_conn( item );

      return 1;
   }

   if ( utf8_strlen( &item->buffer[ item->offset ], &tru_len ) > MAX_NAME || tru_len > MAX_NAME * 4 )
   {
      if ( queue_data( item, NULL, long_msg, long_len ) || queue_data( item, NULL, try_msg, try_len ))
         remove_conn( item );

      return 1;
   }

   return 0;
}

void process_name( struct ccb *item )
{
   int wel_len;
   static int help_len = 0;
   char *ptr, buffer[ 512 ],
        *wel_msg   = "Hello <b>%s</b>.",
        *help_msg  = "Type <b>.help</b> for command list.";

   if ( ! help_len )
   {
      ptr = help_msg;

      while( *ptr++ )
         ++help_len;
   }

   /*
    * Set room temporarily.
    */

   if ( ! item->roomlen && item->buffer[ item->offset ] == '.' )
   {
      item->roomlen = snprintf( item->room, sizeof( item->room ), "%s", &item->buffer[ item->offset + 1 ] );

      if ( !( item->roomlen <= 0 || item->roomlen > ROOMSIZE ))
         return;

      item->roomlen = 0;
      *item->room = '\0';
   }

   if ( check_name( item ))
      return;

   /*
    * Accept name.
    */

   bcopy( &item->buffer[ item->offset ], item->name, item->len );
   item->name[ item->len ] = '\0';
   item->namelen = item->len;

   wel_len = snprintf( buffer, sizeof( buffer ), wel_msg, item->name );

   if ( wel_len < 0 || wel_len >= sizeof( buffer ) || queue_data( item, NULL, buffer, wel_len ))
   {
      remove_conn( item );
      return;
   }

   if ( ! item->roomlen )
      room( item, 0, NULL );
   else
   {
      /*
       * Clear out room so that leave() will not be invoked by go().
       * Requested room may not exist.
       */

      item->total = snprintf( buffer, sizeof( buffer ), "%s", item->room );

      if ( item->total < 0 || item->total >= sizeof( buffer ))
      {
         remove_conn( item );
         return;
      }

      *item->room = '\0';
      item->roomlen = 0;

      go( item, item->total, buffer );
   }

   if ( queue_data( item, NULL, help_msg, help_len ))
      remove_conn( item );
}

void add_conn( int new )
{
   struct ccb *ptr;
   static int full_len = 0, ban_len = 0, chs_len = 0;
   char *chs_msg   = "<b>Enter your handle for this session:</b>",
        *full_msg = "The Public House is full at the moment. Sorry.",
        *ban_msg  = "<b>The Public House " VERSION ".</b>";

   if ( conn_count == max_conn )
   {
      if ( ! full_len )
      {
         char *ptr = full_msg;

         while( *ptr++ )
            ++full_len;
      }

      dprintf( new, "\x81%c%s\x88", full_len, full_msg );
      write( new, "\x0", 1 );
      close( new );
      return;
   }

   if ( ! ban_len )
   {
      char *mptr = ban_msg;

      while( *mptr++ )
         ++ban_len;

      mptr = chs_msg;

      while( *mptr++ )
         ++chs_len;
   }

   dprintf( new, "\x81%c%s", ban_len, ban_msg );
   ++conn_count;

   ptr = stack_pop( free_conns );
   bzero( ptr, sizeof( struct ccb ));

   if (( ptr->buffer = memory( 4 )) == NULL )
   {
      --conn_count;
      stack_push( free_conns, ptr );
      close( new );
      return;
   }

   ptr->timestamp = time( NULL );
   ptr->sock = new;
   ptr->stage = -1;

   ev_set( ptr->sock, EVFILT_READ, EV_ADD | EV_ENABLE, ptr );
   ev_set( ptr->sock, EVFILT_WRITE, EV_ADD | EV_DISABLE, ptr );

   dprintf( new, "\x81%c%s", chs_len, chs_msg );
}

void say_goodbye( struct ccb *item )
{
   char *msg = "<b>Inactivity Timeout.</b></p><p>"
               "<span class=\"indent\">&#8212; <a href=\"javascript:house.connect( '%s', '%s' );"
               "\">Click here to reconnect</a>.</span><br/>";
   char buffer[ 512 ];

   item->len = snprintf( buffer, sizeof( buffer ), msg,
                         ( item->roomlen ? item->room : "" ), ( item->namelen ? item->name : "" ));

   if ( item->len < 0 || item->len >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->len ))
   {
      remove_conn( item );
      return;
   }

   /*
    * Kick him out of the room now.  The close flag is sufficient to
    * prevent any further messages being queued for this client, but we
    * do not want him listed as in the room for future .who queries.
    */

   item->closed = 1;
   leave( item, 0, NULL );
}

void remove_conn( struct ccb *item )
{
   char bye[] = { 0x88, 0 };

   ++item->closed;
   leave( item, 0, NULL );
   write( item->sock, bye, 2 );

   --conn_count;

   closed = item->sock;
   close( item->sock );
   item->sock = 0;

   while( item->first != NULL )
   {
      item->last = item->first->next;

      if ( ! --item->first->buffer->refcount )
      {
         free( item->first->buffer->buffer );
         free( item->first->buffer );
      }

      free( item->first );
      item->first = item->last;
   }

   if ( item->buffer != NULL )
      free( item->buffer );

   bzero( item, sizeof( struct ccb ));
   STACK_PUSH( free_conns, item )
}

void accept_connection()
{
   int conn, n;

   for( n = 0; n < NEWCONS; ++n )
   {
      conn = accept( fd, NULL, 0 );

      if ( conn < 0 )
      {
         if ( errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
         {
            syslog( LOG_ERR, "accept(): %m" );
            exit( 1 );
         }

         return;
      }

      add_conn( conn );
   }
}

void transfer_out( struct ccb *item )
{
   int count;
   struct queue *q;

   if ( item->first == NULL )
      return;

   if (( count = write( item->sock, &item->first->buffer->buffer[ item->first->count ], item->first->total )) < 0 )
   {
      syslog( LOG_ERR, "write(): %m" );
      remove_conn( item );
      return;
   }

   item->first->total -= count;
   item->first->count += count;

   if ( ! item->first->total )
   {
      q = item->first;
      item->first = item->first->next;

      if ( ! --q->buffer->refcount )
      {
         free( q->buffer->buffer );
         free( q->buffer );
      }

      free( q );
      --item->qlen;

      if ( item->first == NULL )
      {
         if ( item->closed )
            remove_conn( item );
         else
         {
            item->last = NULL;
            ev_set( item->sock, EVFILT_WRITE, EV_DISABLE, item );
         }
      }
   }
}

void enter_room( struct ccb *item, char *room )
{
   char buffer[ 512 ];

   if ( item->roomlen )
      leave( item, 0, NULL );

   if (( item->roomlen = snprintf( item->room, sizeof( item->room ), "%s", room )) < 0 )
   {
      remove_conn( item );
      return;
   }
      
   item->total = snprintf( buffer, sizeof( buffer ), "You have <b>entered</b> room <b>%s</b>.", item->room );

   if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
   {
      remove_conn( item );
      return;
   }

   if ( insert_room( item ))
   {
      remove_conn( item );
      return;
   }

   queue_enter_exit_messages( item, 1 );
   who( item, 0, NULL );
}

void go( struct ccb *item, int len, char *rm )
{
   int occ = 0, lock = 0;
   char buffer[ 512 ], roombuf[ ROOMSIZE + 1 ], *ptr1, *ptr2;

   ptr1 = rm;
   ptr2 = item->room;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
   {
      room( item, 0, NULL );
      return;
   }

   /*
    * snprintf() into local buffer to avoid copying into item->buffer from
    * item->buffer.
    */

   if (( len = snprintf( roombuf, sizeof( roombuf ), "%s", rm )) > ROOMSIZE )
      len = ROOMSIZE;

   if ( len < 0 )
   {
      syslog( LOG_ERR, "snprintf(): %m" );
      remove_conn( item );
      return;
   }

   lookup_occupants( roombuf, &occ, &lock, 0 );

   if ( ! occ )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "Room <b>%s</b> is not open.", roombuf );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   if ( lock )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "Room <b>%s</b> is <b>locked</b>.", roombuf );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   if ( occ >= max_occupants )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "Room <b>%s</b> is <b>full</b>.", roombuf );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   if ( item->roomlen )
      leave( item, 0, NULL );

   enter_room( item, roombuf );
}

char *make_name()
{
   static char buffer[ ROOMSIZE + 1 ];
   char *ptr;
   int i, j;

   for( ptr = buffer, i = 0; i < sizeof( buffer ) - 1; ++i )

      if ( !( i % 2 ))
      {
         do
         {
            j = arc4random_uniform( 26 );
            *ptr = j + 97;
         }
         while( *ptr == 97 || *ptr == 101 || *ptr == 105 || *ptr == 111 || *ptr == 117 );

         ++ptr;
      }
      else
         switch( j = arc4random() % 5 )
         {
            case 0:
               *ptr++ = 97;
               break;

            case 1:
               *ptr++ = 101;
               break;

            case 2:
               *ptr++ = 105;
               break;

            case 3:
               *ptr++ = 111;
               break;

            case 4:
               *ptr++ = 117;
         }

   buffer[ i ] = '\0';
   return buffer;
}

void make( struct ccb *item, int len, char *room )
{
   int occ = 0, lock = 0;
   char buffer[ 512 ];

AGAIN:
   room = make_name();
   lookup_occupants( room, &occ, &lock, 0 );

   if ( occ )
      goto AGAIN;

   item->total = snprintf( buffer, sizeof( buffer ),
                           "Room <b>%s</b> has been <b>opened</b>.",
                           room );

   if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
   {
      remove_conn( item );
      return;
   }

   enter_room( item, room );
}

void room( struct ccb *item, int len, char *ignored )
{
   char buffer[ 128 ];

   item->total = snprintf( buffer, sizeof( buffer ), "You are in %s <b>%s</b>.",
                           ( item->roomlen ? "room" : "the" ), ( item->roomlen ? item->room : "foyer" ));

   if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
      remove_conn( item );
}

void queue_enter_exit_messages( struct ccb *item, int enter )
{
   char buffer[ 512 ], *msg = "<b>%s</b> has <b>%s</b> the room.";
   int len;

   len = snprintf( buffer, sizeof( buffer ), msg, item->name, ( enter ? "entered" : "exited" ));

   if ( len < 0 || len >= sizeof( buffer ) || queue_data( item, lookup_occupants( item->room, NULL, NULL, 0 ), buffer, len ))
      remove_conn( item );
}

void leave( struct ccb *item, int len, char *ignored )
{
   char buffer[ 512 ];

   if ( ! item->roomlen )
   {
      if ( ! item->closed )
      {
         item->total = snprintf( buffer, sizeof( buffer ),
                                 "<span class=\"indent\">&#8212; <a href=\"javascript:house.stop()\">Click here to exit the chat system</a>.</span>" );

         if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
            remove_conn( item );
      }

      return;
   }

   /*
    * We must queue the messages before we remove him, or the messages will
    * be mistakenly queued for him if there are no other roomies.
    */

   queue_enter_exit_messages( item, 0 );
   remove_room( item );

   if ( ! item->closed )
   {
      int occ = 0;

      item->total = snprintf( buffer, sizeof( buffer ), "You have <b>exited</b> room <b>%s</b>.", item->room );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
      {
         remove_conn( item );
         return;
      }

      lookup_occupants( item->room, &occ, NULL, 0 );

      if ( ! occ )
      {
         item->total = snprintf( buffer, sizeof( buffer ), "Room <b>%s</b> is <b>closed</b>.", item->room );

         if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         {
            remove_conn( item );
            return;
         }
      }
   }

   *item->room = '\0';
   item->roomlen = 0;

   if ( ! item->closed )
      room( item, 0, NULL );
}

void help( struct ccb *item, int len, char *ignored )
{
   char buffer[ 2048 ];

   len = snprintf( buffer, sizeof( buffer ), "%s",
      "Type <b>.open</b> to open a room.<br/>\n"
      "Type <b>.go &lt;room&gt;</b> to enter an open room.<br/>\n"
      "Type <b>.room</b> to find out what room you are in.<br/>\n"
      "Type <b>.who</b> to find out who else is in your room.<br/>\n"
      "Type <b>.lock</b> to lock a room.<br/>\n"
      "Type <b>.unlock</b> to unlock a room.<br/>\n"
      "Type <b>.exit</b> to return to the foyer.<br/>\n"
      "Type <b>.help</b> to display this summary again.\n" );

   if ( len < 0 || len >= sizeof( buffer ) || queue_data( item, NULL, buffer, len ))
      remove_conn( item );
}

void who( struct ccb *item, int len, char *ignored )
{
   char buffer[ 512 ];
   struct list *ptr;
   int whoodles;

   if ( ! item->roomlen )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "%s", "You are not permitted to know who is in the <b>foyer</b>." );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   for( ptr = lookup_occupants( item->room, &whoodles, NULL, 0 ); ptr != NULL; ptr = ptr->next )
      if ( ptr->conn != item )
      {
         item->total = snprintf( buffer, sizeof( buffer ), "<b>%s</b> is here.", ptr->conn->name );

         if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         {
            remove_conn( item );
            return;
         }
      }

   if ( ! --whoodles )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "%s", "You are <b>alone</b> in this room." );

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );
   }
}

void queue_lock_unlock_messages( struct ccb *item, int lock )
{
   char buffer[ 512 ], *lock_msg = "<b>%s</b> has <b>%s</b> the room.";
   int len;

   len = snprintf( buffer, sizeof( buffer ), lock_msg, item->name, ( lock ? "locked" : "unlocked" ));

   if ( len < 0 || len >= sizeof( buffer ) || queue_data( item, lookup_occupants( item->room, NULL, NULL, 0 ), buffer, len ))
      remove_conn( item );
}

void lock_unlock( struct ccb *item, int lock )
{
   char buffer[ 512 ];

   if ( ! item->roomlen )
   {
      item->total = snprintf( buffer, sizeof( buffer ), "You cannot %s the <b>foyer</b>.", ( lock ? "lock" : "unlock" ));

      if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   lookup_occupants( item->room, NULL, NULL, ( lock ? 1 : -1 ));

   item->total = snprintf( buffer, sizeof( buffer ), "Room <b>%s</b> is <b>%s</b>.",
                           item->room, ( lock ? "locked" : "unlocked" ));

   if ( item->total < 0 || item->total >= sizeof( buffer ) || queue_data( item, NULL, buffer, item->total ))
   {
      remove_conn( item );
      return;
   }

   queue_lock_unlock_messages( item, lock );
}

void lock( struct ccb *item, int len, char *ignored )
{
   lock_unlock( item, 1 );
}

void unlock( struct ccb *item, int len, char *ignored )
{
   lock_unlock( item, 0 );
}

void process_user_input( struct ccb *item )
{
   char *strings[] = { ".go ",  ".open", ".room", ".who", ".exit", ".lock",  ".unlock", ".help", NULL };
   void ( *func[] )( struct ccb *, int, char * ) = { go, make, room,  who, leave, lock, unlock, help, NULL };

   char buffer[ iobufsize ], *ptr1, *ptr2, **ptr;
   int i, t;

   for( i = 0, ptr = strings; *ptr != NULL; ++i, ++ptr )
   {
      t = item->len;
      ptr1 = &item->buffer[ item->offset ];
      ptr2 = *ptr;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         --t;
         ++ptr1;
         ++ptr2;
      }

      if ( i )
      {
         if ( ! *ptr1 && ! *ptr2 )
         {
            func[ i ]( item, t, ptr1 );
            return;
         }
      }
      else if ( *ptr1 && ! *ptr2 )
      {
         func[ i ]( item, t, ptr1 );
         return;
      }
   }

   if ( item->buffer[ item->offset ] == '.' )
   {
      item->total = snprintf( buffer, iobufsize, "%s", "Unrecognized command." );

      if ( item->total < 0 || item->total >= iobufsize || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   if ( ! item->roomlen )
   {
      item->total = snprintf( buffer, iobufsize, "%s", "You cannot chat in the <b>foyer</b>." );

      if ( item->total < 0 || item->total >= iobufsize || queue_data( item, NULL, buffer, item->total ))
         remove_conn( item );

      return;
   }

   t = snprintf( buffer, iobufsize, "<b>%s</b>: <span class=\"chat\">%s</span>", item->name, &item->buffer[ item->offset ] );

   if ( t < 0 || t >= iobufsize || queue_data( NULL, lookup_occupants( item->room, NULL, NULL, 0 ), buffer, t ))
      remove_conn( item );
}

int verify_opcode_and_payload_size( struct ccb *item )
{
   int i;

   /*
    * Check for bad opcode.  Opcodes are close, ping, pong, text, and fragment.
    */

   if (( i = ( item->buffer[ 0 ] & 0x0f )) && i != 0x08 && i != 0x09 && i != 0x0a && i != 0x01 && i != 0x00 )
      return 1;

   /*
    * Client is closing connection.
    */

   if (( unsigned char )item->buffer[ 0 ] == 0x88 )
      return 1;

   /*
    * Frame too long.
    */

   if (( item->len = (( unsigned char )item->buffer[ 1 ] & 0x7f )) > 126 )
      return 1;

   /*
    * Payload without masking key.
    */

   if ( item->len && ! (( unsigned char )item->buffer[ 1 ] & 0x80 ))
      return 1;

   return 0;
}

int process_payload( struct ccb *item )
{
   int ping, i;
   unsigned char *ptr;

   /*
    * Accept but discard incoming data on "closed" connection.
    * Connection will be closed when all pending outgoing data has been delivered.
    */

   if ( item->closed )
      return 0;

   ping = (( unsigned char )item->buffer[ 0 ] == 0x89 );

   /*
    * Unmask payload.  Overwrite HTML-significant characters.  Ptr must
    * point to unsigned chars so that UTF-8 bytes with high-order bit set
    * are not interpreted as < 32.
    */

   for( i = 0, ptr = ( unsigned char *)&item->buffer[ item->offset ]; i < item->len; ++ptr, ++i )
   {
      *ptr ^= item->mask[ i % 4 ];

      if ( ! ping && ( *ptr < 32 || *ptr == '&' || *ptr == '<' || *ptr == '>' ))
         *ptr = '_';
   }

   item->buffer[ item->len + item->offset ] = '\0';

   if ( ping )
   {
      if ( queue_pong( item ))
         return 1;

      return 0;
   }

   item->timestamp = time( NULL );

   /*
    * If not FIN, discard fragment, close connection.
    */

   if ( !( item->buffer[ 0 ] & 0x80 ))
      return 1;

   if ( ! item->namelen )
      process_name( item );
   else
      process_user_input( item );

   return 0;
}

void transfer_in( struct ccb *item )
{
   int i;

   switch( item->stage )
   {
      /*
       * Read and discard the newline character that terminates
       * the empty Cookie header from Prospero. Guaranteed to be ready because
       * we are processing a read event.
       */

      case -1:
         if ( read( item->sock, item->buffer, 1 ) <= 0 )
         {
            remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read first byte of header.
       */

      case 0:

         if (( i = read( item->sock, item->buffer, 1 )) <= 0 )
         {
            if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
               return;

            remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read second byte of header.
       */

      case 1:

         if (( i = read( item->sock, &item->buffer[ 1 ], 1 )) <= 0 )
         {
            if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
               return;

            remove_conn( item );
            return;
         }

         if ( verify_opcode_and_payload_size( item ))
         {
            remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read first byte of extended length.
       */

      case 2:

         if ( item->len == 126 && ( i = read( item->sock, &item->buffer[ 2 ], 1 )) <= 0 )
         {
            if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
               return;

            remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read second byte of extended length.  Verify payload will fit in buffer with
       * annotation.
       */

      case 3:

         if ( item->len == 126 )
         {
            if (( i = read( item->sock, &item->buffer[ 3 ], 1 )) <= 0 )
            {
               if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
                  return;

               remove_conn( item );
               return;
            }

            item->len = (( unsigned char )item->buffer[ 2 ] << 8 ) + ( unsigned char )item->buffer[ 3 ];

            if ( item->len > iobufsize - GAP )
            {
               remove_conn( item );
               return;
            }
         }

         ++item->stage;
         item->total = 4;
         item->offset = 0;

      /*
       * Read masking key.
       */

      case 4:

         if (( i = read( item->sock, &item->mask[ item->offset ], item->total )) <= 0 )
         {
            if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
               return;

            remove_conn( item );
            return;
         }

         item->offset += i;
         item->total -= i;

         if ( item->total )
            return;

         if ( ! ( item->total = item->len ))
         {
            item->stage = 0;

            /*
             * Ignore unsolicited empty payload pongs.
             * Return pongs for empty payload pings.
             */

            if (( unsigned char )item->buffer[ 0 ] == 0x89 )
               queue_pong( item );

            return;
         }

         item->offset = ( item->len > 125 ? 4 : 2 );

         if (( item->buffer = realloc( item->buffer, item->offset + item->total + 1 )) == NULL )
         {
            remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read payload.
       */

      case 5:

         if (( i = read( item->sock, &item->buffer[ item->offset ], item->total )) <= 0 )
         {
            if ( i < 0 && ( errno == EWOULDBLOCK || errno == EAGAIN ))
               return;

            remove_conn( item );
            return;
         }

         item->offset += i;
         item->total -= i;

         if ( item->total )
            return;

         item->stage = 0;
         item->offset = ( item->len > 125 ? 4 : 2 );

         /*
          * Ignore unsolicited pong per RFC.
          */

         if (( unsigned char )item->buffer[ 0 ] == 0x8a )
            return;

         if ( process_payload( item ))
            remove_conn( item );
   }
}

/*
 * If we have dropped a connection, we traverse the output kqueue and mark
 * any events for that connection as invalid by setting the socket
 * descriptor to 0.
 */

void remove_events( struct ccb *item, int n )
{
   struct kevent *eptr;

   for( eptr = &outqueue[ ++n ]; n < out; ++n, ++eptr )
      if ( eptr->udata == item )
         eptr->ident = 0;
}

void sigterm_handler( int signo )
{
   ++sigterm;
}

void set_sigterm_intr()
{
   struct sigaction sigact;

   sigact.sa_handler = sigterm_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGTERM, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction(): %m" );
}

void drop_connections()
{
   struct ccb *ptr;
   time_t t;
   int i;

   t = time( NULL );

   for( i = 0, ptr = conns; i < max_conn; ++i, ++ptr )
      if ( ptr->sock && ! ptr->closed && ( t - ptr->timestamp ) > seconds )
         say_goodbye( ptr );
}

void process_clients()
{
   int kq, n;
   struct kevent *eptr;

   qlen = max_conn * MAX_EVENT;

   if (( inqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( inqueue, sizeof( struct kevent ) * qlen );
   
   if (( outqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( outqueue, sizeof( struct kevent ) * qlen );

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   ev_set( fd, EVFILT_READ, EV_ADD | EV_ENABLE, NULL );
   set_timer();

   for( ; ; )
   {
      if ( sigterm )
         break;

      set_sigterm_intr();
      out = kevent( kq, inqueue, in, outqueue, qlen, NULL );
      in = 0;

      if ( sigterm )
         break;

      if ( out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %m" );
         exit( 1 );
      }

      signal( SIGTERM, sigterm_handler );

      for( n = 0, eptr = &outqueue[ n ]; n < out; ++n, ++eptr )
      {
         struct ccb *item;

         if ( ! eptr->ident || eptr->flags & EV_ERROR )
            continue;

         closed = 0;
         item = ( struct ccb *)eptr->udata;

         if ( eptr->filter == EVFILT_TIMER )
            drop_connections();
         else if ( eptr->ident == fd )
            accept_connection();
         else if ( eptr->filter == EVFILT_READ )
            transfer_in( item );
         else
            transfer_out( item );

         if ( closed )
            remove_events( item, n );
      }
   }
}

void set_hostname()
{
   long len;
   char *ptr1, *ptr2, *host;

   if (( len = sysconf( _SC_HOST_NAME_MAX )) < 0 )
   {
      syslog( LOG_ERR, "sysconf(): %m" );
      len = PATH_MAX + 1;
   }

   ++len;

   if (( host = memory( len )) == NULL )
      exit( 1 );

   *host = '\0';

   if ( gethostname( host, len ))
   {
      syslog( LOG_ERR, "gethostname(): %m" );
      exit( 1 );
   }

   ptr1 = "localhost";
   ptr2 = host;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( *ptr1 || *ptr2 )
      hostname = host;
   else
   {
      free( host );
      if (( hostname = str_dup( "127.0.0.1", 0 )) == NULL )
         exit( 1 );
   }
}

void init( int argc, char **argv )
{
   struct ccb *ptr;
   int i;

   signal( SIGPIPE, SIG_IGN );
   signal( SIGTERM, sigterm_handler );

   set_options( argc, argv );

   if ( ! *hostname )
      set_hostname();

   if (( conns = memory( sizeof( struct ccb ) * max_conn )) == NULL )
      exit( 1 );

   bzero( conns, sizeof( struct ccb ) * max_conn );

   if (( free_conns = make_stack()) == NULL )
      exit( 1 );

   for( i = 0, ptr = conns; i < max_conn; ++i, ++ptr )
      STACK_PUSH( free_conns, ptr )

   hash_size = max_conn / 4;

   if (( hash = memory( sizeof( struct hash * ) * hash_size )) == NULL )
      exit( 1 );

   bzero( hash, sizeof( struct hash * ) * hash_size );

   openlog( "house", LOG_PID, LOG_DAEMON );
   logging = 1;
}

int main( int argc, char **argv )
{
   init( argc, argv );

   if ( ! testing )
      become_daemon();

   start_listening_unix();
   change_identity();
   process_clients();

   return 0;
}
